# Design Document: Dimension-Based Inventory System

## Overview

The dimension-based inventory system transforms Raw Material (RM) tracking from simple quantity-based to sophisticated dimension-aware management. This system enables clothing manufacturers to track fabric pieces by their physical dimensions (length × width), manage waste efficiently, and optimize material utilization through intelligent allocation algorithms.

The system maintains the existing UUID-based tracking for Finished Goods (FG) while implementing a parallel dimension-based tracking system exclusively for Raw Materials. This dual approach ensures optimal tracking methods for each material type.

## Architecture

### High-Level Architecture

```mermaid
graph TB
    subgraph "Frontend Layer"
        A[GRN Form] --> B[BOM Form]
        B --> C[Production Interface]
        C --> D[Inventory Dashboard]
    end
    
    subgraph "API Layer"
        E[Procurement API] --> F[Production API]
        F --> G[Inventory API]
        G --> H[Reporting API]
    end
    
    subgraph "Service Layer"
        I[GRN Service] --> J[Material Allocation Service]
        J --> K[Cutting Operations Service]
        K --> L[Waste Management Service]
        L --> M[Inventory Calculation Service]
    end
    
    subgraph "Data Layer"
        N[rm_inventory_pieces] --> O[rm_cutting_operations]
        O --> P[Enhanced BOM Items]
        P --> Q[Enhanced GRN Items]
        Q --> R[Enhanced Products]
    end
    
    A --> E
    E --> I
    I --> N
```

### System Integration Points

1. **Procurement Integration**: GRN processing creates dimension-based inventory pieces
2. **Production Integration**: BOM validation and material allocation use dimensions
3. **Inventory Integration**: Dual-layer tracking with dimension summaries
4. **Reporting Integration**: Material utilization and waste analytics

## Components and Interfaces

### 1. Enhanced Product Model

**Purpose**: Add dimension tracking capabilities to existing Product model

**New Fields**:
- `unit_of_measure`: ENUM('inch', 'cm', 'm') - Default unit for the product (dimension-based RM only)
- `track_by_dimensions`: BOOLEAN - TRUE for dimension-based RM, FALSE for special items RM

**Business Rules**:
- **Dimension-Based RM** (fabric, leather, etc.):
  - Automatic enforcement: `product_type = 'RM'` and dimension-based → `track_by_dimensions = TRUE`
  - Validation prevents disabling dimension tracking for dimension-based RM products
  - Unit of measure can be overridden per inventory piece
- **Special Items RM** (zippers, buttons, threads, etc.):
  - `track_by_dimensions = FALSE` (quantity-based tracking)
  - Use standard inventory tracking (not dimension-based)
  - Examples: Zippers, buttons, snaps, threads, labels, tags, hooks

### 2. RM Inventory Pieces Model

**Purpose**: Track individual pieces of Raw Materials with dimensions

```javascript
// Model Structure
{
  id: BIGINT PRIMARY KEY,
  product_id: BIGINT NOT NULL,
  grn_item_id: BIGINT NULL,
  piece_number: INT NOT NULL,
  
  // Dimensions
  length: DECIMAL(12, 3) NOT NULL,
  width: DECIMAL(12, 3) NOT NULL,
  unit: ENUM('inch', 'cm', 'm') NOT NULL,
  
  // Status tracking
  status: ENUM('FULL', 'USABLE', 'WASTE', 'SCRAP') NOT NULL,
  
  // Current usable dimensions
  usable_length: DECIMAL(12, 3) NULL,
  usable_width: DECIMAL(12, 3) NULL,
  
  // Scrap dimensions
  scrap_length: DECIMAL(12, 3) DEFAULT 0,
  scrap_width: DECIMAL(12, 3) DEFAULT 0
}
```

**Key Operations**:
- `createPieces(grnItem, dimensions)`: Create pieces from GRN processing
- `findSuitablePieces(requirements)`: Find pieces matching BOM requirements
- `updateAfterCutting(pieceId, cutDimensions)`: Update piece after cutting operation
- `markAsScrap(pieceId, scrapDimensions)`: Mark portions as scrap

### 3. RM Cutting Operations Model

**Purpose**: Track cutting operations and resulting material changes

```javascript
// Model Structure
{
  id: BIGINT PRIMARY KEY,
  production_order_id: BIGINT NOT NULL,
  rm_piece_id: BIGINT NOT NULL,
  bom_item_id: BIGINT NOT NULL,
  
  // Cut dimensions
  cut_length: DECIMAL(12, 3) NOT NULL,
  cut_width: DECIMAL(12, 3) NOT NULL,
  unit: ENUM('inch', 'cm', 'm') NOT NULL,
  
  // Results
  remaining_piece_id: BIGINT NULL,
  waste_pieces: JSON NULL,
  scrap_dimensions: JSON NULL,
  
  cut_by_user_id: BIGINT NULL,
  cut_at: DATETIME DEFAULT CURRENT_TIMESTAMP
}
```

### 4. Enhanced BOM Items Model

**Purpose**: Add dimension requirements to existing BOM structure

**New Fields**:
- `required_length`: DECIMAL(12, 3) - Required length per unit
- `required_width`: DECIMAL(12, 3) - Required width per unit
- `dimension_unit`: ENUM('inch', 'cm', 'm') - Unit for dimensions
- `use_dimensions`: BOOLEAN - Flag for dimension-based BOM items

**Validation Rules**:
- RM products in BOM must have `use_dimensions = TRUE`
- Dimension fields required when `use_dimensions = TRUE`
- Unit compatibility validation with available materials

### 5. Enhanced GRN Items Model

**Purpose**: Add dimension fields to procurement processing

**New Fields**:
- `piece_length`: DECIMAL(12, 3) - Length of each piece
- `piece_width`: DECIMAL(12, 3) - Width of each piece
- `dimension_unit`: ENUM('inch', 'cm', 'm') - Unit for dimensions
- `pieces_count`: INT - Number of pieces with these dimensions

**Processing Logic**:
- Create `pieces_count` number of `rm_inventory_pieces` records
- Each piece gets unique `piece_number` within the GRN
- Calculate total area: `piece_length × piece_width × pieces_count`

### 6. Material Allocation Service

**Purpose**: Intelligent allocation of RM pieces to production orders

**Core Algorithm**:
```javascript
function allocateMaterials(bomRequirements, productionQuantity) {
  const allocations = [];
  
  for (const requirement of bomRequirements) {
    if (requirement.use_dimensions) {
      // Find suitable pieces in priority order
      const suitablePieces = findPiecesInOrder([
        'FULL',   // Prefer full pieces first
        'USABLE', // Then usable pieces
        'WASTE'   // Finally waste pieces if suitable
      ], requirement);
      
      // Allocate pieces with unit conversion
      const allocation = allocateWithConversion(
        suitablePieces, 
        requirement, 
        productionQuantity
      );
      
      allocations.push(allocation);
    }
  }
  
  return allocations;
}
```

**Key Features**:
- Priority-based allocation (FULL → USABLE → WASTE)
- Automatic unit conversion between inches, cm, meters
- Waste piece suggestion for cost optimization
- Insufficient material detection with maximum quantity calculation

### 7. Cutting Operations Service

**Purpose**: Process cutting operations and manage resulting materials

**Cutting Workflow**:
1. **Validate Cut**: Ensure cut dimensions fit within usable dimensions
2. **Execute Cut**: Create cutting operation record
3. **Update Original Piece**: Modify usable dimensions or mark as consumed
4. **Create Waste Pieces**: Generate new pieces for reusable leftovers
5. **Mark Scrap**: Identify and track unusable small pieces

**Waste Classification Logic**:
```javascript
function classifyRemainingMaterial(originalPiece, cutDimensions) {
  const remaining = calculateRemainingDimensions(originalPiece, cutDimensions);
  
  if (remaining.length >= MIN_USABLE_LENGTH && remaining.width >= MIN_USABLE_WIDTH) {
    return createWastePiece(remaining, 'WASTE');
  } else {
    return markAsScrap(remaining);
  }
}
```

### 8. Inventory Calculation Service

**Purpose**: Calculate inventory summaries and availability

**Calculation Methods**:
- **Piece Count**: `COUNT(*) WHERE status IN ('FULL', 'USABLE', 'WASTE')`
- **Total Area**: `SUM(length × width) GROUP BY status`
- **Usable Area**: `SUM(usable_length × usable_width) WHERE status != 'SCRAP'`
- **Waste Area**: `SUM(length × width) WHERE status = 'WASTE'`

**Performance Optimization**:
- Denormalized summary fields in `inventories` table
- Indexed queries on `(product_id, status)` combinations
- Cached calculations updated on material changes

## Data Models

### Database Schema Changes

#### 1. Products Table Enhancement
```sql
ALTER TABLE products 
ADD COLUMN unit_of_measure ENUM('inch', 'cm', 'm') DEFAULT 'm',
ADD COLUMN track_by_dimensions BOOLEAN DEFAULT FALSE;

-- Create index for dimension tracking queries
CREATE INDEX idx_products_dimensions ON products(product_type, track_by_dimensions);
```

#### 2. New RM Inventory Pieces Table
```sql
CREATE TABLE rm_inventory_pieces (
  id BIGINT PRIMARY KEY AUTO_INCREMENT,
  product_id BIGINT NOT NULL,
  grn_item_id BIGINT NULL,
  piece_number INT NOT NULL,
  
  -- Dimensions
  length DECIMAL(12, 3) NOT NULL,
  width DECIMAL(12, 3) NOT NULL,
  unit ENUM('inch', 'cm', 'm') NOT NULL DEFAULT 'm',
  
  -- Status tracking
  status ENUM('FULL', 'USABLE', 'WASTE', 'SCRAP') NOT NULL DEFAULT 'FULL',
  
  -- Current usable dimensions
  usable_length DECIMAL(12, 3) NULL,
  usable_width DECIMAL(12, 3) NULL,
  
  -- Scrap dimensions
  scrap_length DECIMAL(12, 3) DEFAULT 0,
  scrap_width DECIMAL(12, 3) DEFAULT 0,
  
  -- Metadata
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  
  FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
  FOREIGN KEY (grn_item_id) REFERENCES grn_items(id) ON DELETE SET NULL,
  
  INDEX idx_product_status (product_id, status),
  INDEX idx_status (status),
  INDEX idx_dimensions (length, width, unit),
  INDEX idx_usable_dimensions (usable_length, usable_width)
);
```

#### 3. RM Cutting Operations Table
```sql
CREATE TABLE rm_cutting_operations (
  id BIGINT PRIMARY KEY AUTO_INCREMENT,
  production_order_id BIGINT NOT NULL,
  rm_piece_id BIGINT NOT NULL,
  bom_item_id BIGINT NOT NULL,
  
  -- Cut dimensions
  cut_length DECIMAL(12, 3) NOT NULL,
  cut_width DECIMAL(12, 3) NOT NULL,
  unit ENUM('inch', 'cm', 'm') NOT NULL,
  
  -- Results
  remaining_piece_id BIGINT NULL,
  waste_pieces JSON NULL,
  scrap_dimensions JSON NULL,
  
  -- Metadata
  cut_by_user_id BIGINT NULL,
  cut_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  
  FOREIGN KEY (production_order_id) REFERENCES production_orders(id) ON DELETE CASCADE,
  FOREIGN KEY (rm_piece_id) REFERENCES rm_inventory_pieces(id) ON DELETE CASCADE,
  FOREIGN KEY (bom_item_id) REFERENCES bom_items(id) ON DELETE CASCADE,
  FOREIGN KEY (cut_by_user_id) REFERENCES users(id) ON DELETE SET NULL,
  
  INDEX idx_production_order (production_order_id),
  INDEX idx_rm_piece (rm_piece_id),
  INDEX idx_cut_date (cut_at)
);
```

#### 4. Enhanced BOM Items Table
```sql
ALTER TABLE bom_items 
ADD COLUMN required_length DECIMAL(12, 3) NULL,
ADD COLUMN required_width DECIMAL(12, 3) NULL,
ADD COLUMN dimension_unit ENUM('inch', 'cm', 'm') NULL,
ADD COLUMN use_dimensions BOOLEAN DEFAULT FALSE;

-- Create index for dimension-based BOM queries
CREATE INDEX idx_bom_items_dimensions ON bom_items(use_dimensions, required_length, required_width);
```

#### 5. Enhanced GRN Items Table
```sql
ALTER TABLE grn_items 
ADD COLUMN piece_length DECIMAL(12, 3) NULL,
ADD COLUMN piece_width DECIMAL(12, 3) NULL,
ADD COLUMN dimension_unit ENUM('inch', 'cm', 'm') NULL,
ADD COLUMN pieces_count INT DEFAULT 1;

-- Create index for dimension-based GRN queries
CREATE INDEX idx_grn_items_dimensions ON grn_items(piece_length, piece_width, dimension_unit);
```

#### 6. Enhanced Inventories Table
```sql
ALTER TABLE inventories 
ADD COLUMN total_full_area DECIMAL(15, 3) NULL,
ADD COLUMN total_usable_area DECIMAL(15, 3) NULL,
ADD COLUMN total_waste_area DECIMAL(15, 3) NULL,
ADD COLUMN total_scrap_area DECIMAL(15, 3) NULL,
ADD COLUMN dimension_unit ENUM('inch', 'cm', 'm') NULL;
```

### Data Relationships

```mermaid
erDiagram
    products ||--o{ rm_inventory_pieces : "has many"
    grn_items ||--o{ rm_inventory_pieces : "creates"
    rm_inventory_pieces ||--o{ rm_cutting_operations : "cut in"
    production_orders ||--o{ rm_cutting_operations : "performs"
    bom_items ||--o{ rm_cutting_operations : "requires"
    
    products {
        bigint id PK
        string name
        enum product_type
        boolean track_by_dimensions
        enum unit_of_measure
    }
    
    rm_inventory_pieces {
        bigint id PK
        bigint product_id FK
        bigint grn_item_id FK
        int piece_number
        decimal length
        decimal width
        enum unit
        enum status
        decimal usable_length
        decimal usable_width
        decimal scrap_length
        decimal scrap_width
    }
    
    rm_cutting_operations {
        bigint id PK
        bigint production_order_id FK
        bigint rm_piece_id FK
        bigint bom_item_id FK
        decimal cut_length
        decimal cut_width
        enum unit
        json waste_pieces
        json scrap_dimensions
    }
```

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system—essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

Based on the prework analysis, I've identified the following correctness properties that must hold for the dimension-based inventory system:

### Property 1: RM Dimension Validation
*For any* RM product in GRN processing, the system should reject items without piece_length, piece_width, and dimension_unit fields
**Validates: Requirements 1.1, 1.4, 8.2**

### Property 2: Dimension-Based Piece Creation
*For any* GRN with RM items containing dimensions, the system should create exactly pieces_count number of rm_inventory_pieces records
**Validates: Requirements 1.2, 1.3**

### Property 3: Area Calculation Consistency
*For any* set of dimensions and piece count, total area should equal length × width × pieces_count
**Validates: Requirements 1.6**

### Property 4: BOM Dimension Requirements
*For any* RM product in BOM, the system should require use_dimensions = TRUE with required_length and required_width
**Validates: Requirements 2.1, 2.2, 8.3**

### Property 5: Material Allocation Priority
*For any* production order with RM requirements, the system should prioritize allocation in order: FULL pieces, then USABLE pieces, then WASTE pieces
**Validates: Requirements 3.2, 5.1**

### Property 6: Dimension Sufficiency Check
*For any* material allocation, selected pieces should have usable dimensions greater than or equal to BOM requirements
**Validates: Requirements 3.1, 3.4**

### Property 7: Unit Conversion Accuracy
*For any* dimension matching between different units, converted values should maintain accuracy to 3 decimal places
**Validates: Requirements 3.5, 9.1, 9.2**

### Property 8: Cutting Operation Audit Trail
*For any* production confirmation using RM, the system should create cutting operation records with exact cut dimensions
**Validates: Requirements 4.1, 11.1**

### Property 9: Waste Status Management
*For any* cutting operation that creates reusable material, the system should create waste pieces with status WASTE
**Validates: Requirements 4.3, 5.2**

### Property 10: Scrap Prevention
*For any* production allocation attempt, the system should reject pieces with status SCRAP
**Validates: Requirements 5.5, 10.5**

### Property 11: Inventory Calculation Accuracy
*For any* RM product, inventory quantity should equal the count of pieces with status IN ('FULL', 'USABLE', 'WASTE')
**Validates: Requirements 6.1, 6.2**

### Property 12: Multi-Dimension Independence
*For any* product received in multiple dimension sets, each dimension set should create separate inventory pieces while linking to the same product
**Validates: Requirements 7.1, 7.2**

### Property 13: RM Dimension Enforcement
*For any* dimension-based RM product creation, the system should automatically set track_by_dimensions to TRUE and prevent disabling
*For any* special items RM product creation, the system should set track_by_dimensions to FALSE (quantity-based)
**Validates: Requirements 8.1, 8.5, 13.2**

### Property 14: Fractional Dimension Support
*For any* dimension input with decimal values, the system should accept and process fractional dimensions correctly
**Validates: Requirements 9.4**

### Property 15: Scrap Dimension Separation
*For any* piece with scrap dimensions, the system should track scrap separately from usable dimensions
**Validates: Requirements 10.2, 10.3**

### Property 16: Cutting Traceability
*For any* cutting operation, all resulting pieces should link back to the original cutting operation
**Validates: Requirements 11.4**

### Property 17: Waste Reuse Suggestion
*For any* production planning with available waste pieces, the system should suggest waste pieces that match BOM requirements
**Validates: Requirements 5.3, 12.4**

### Property 18: Material Utilization Calculation
*For any* cutting operation, material utilization efficiency should be calculated as (used_area / total_area) × 100
**Validates: Requirements 11.5, 12.1**

## Error Handling

### Validation Errors

1. **Missing Dimensions**: Clear error messages when RM products lack dimension data
2. **Unit Mismatch**: Validation errors for incompatible units between BOM and inventory
3. **Insufficient Materials**: Detailed messages showing available vs. required dimensions
4. **Invalid Cuts**: Prevention of cuts larger than available material dimensions

### Business Logic Errors

1. **Allocation Failures**: Graceful handling when no suitable pieces are found
2. **Conversion Errors**: Fallback handling for unit conversion edge cases
3. **Concurrent Access**: Optimistic locking for piece allocation conflicts
4. **Data Integrity**: Transaction rollback for failed cutting operations

### Recovery Mechanisms

1. **Partial Allocation**: Allow partial production when some materials are available
2. **Alternative Suggestions**: Recommend alternative pieces or production quantities
3. **Waste Utilization**: Automatic suggestion of waste pieces for cost optimization
4. **Audit Trail**: Complete tracking for debugging and compliance

## Testing Strategy

### Dual Testing Approach

The testing strategy employs both unit testing and property-based testing to ensure comprehensive coverage:

**Unit Tests**: Focus on specific examples, edge cases, and integration points
- Specific dimension scenarios (e.g., Cotton Fabric 2m×2m×5pcs)
- Edge cases (very small dimensions, unit boundaries)
- Error conditions (missing dimensions, invalid units)
- Integration between GRN processing and inventory creation

**Property Tests**: Verify universal properties across all inputs
- Dimension validation for all RM products
- Area calculations with random dimensions and quantities
- Material allocation with various piece configurations
- Cutting operations with different dimension combinations

### Property-Based Testing Configuration

- **Minimum 100 iterations** per property test for thorough coverage
- **Test Library**: Use `fast-check` for JavaScript property-based testing
- **Tag Format**: Each test tagged as **Feature: dimension-based-inventory, Property {number}: {property_text}**
- **Generator Strategy**: Smart generators that create realistic dimension ranges and valid unit combinations

### Test Data Generators

```javascript
// Example property test generators
const dimensionGen = fc.record({
  length: fc.float({ min: 0.1, max: 100.0 }),
  width: fc.float({ min: 0.1, max: 100.0 }),
  unit: fc.constantFrom('inch', 'cm', 'm')
});

const dimensionBasedRMGen = fc.record({
  name: fc.string({ minLength: 1, maxLength: 50 }),
  product_type: fc.constant('RM'),
  track_by_dimensions: fc.constant(true),
  unit_of_measure: fc.constantFrom('inch', 'cm', 'm')
});

const specialItemsRMGen = fc.record({
  name: fc.string({ minLength: 1, maxLength: 50 }),
  product_type: fc.constant('RM'),
  track_by_dimensions: fc.constant(false),
  unit_of_measure: fc.constant(null)
});
```

### Integration Testing

- **End-to-end workflows**: Complete GRN → BOM → Production → Waste cycles
- **Multi-dimension scenarios**: Same product in various dimension combinations
- **Waste reuse flows**: Allocation of waste pieces to subsequent production orders
- **Unit conversion accuracy**: Cross-unit material matching and allocation

### Performance Testing

- **Large inventory queries**: Performance with thousands of dimension pieces
- **Complex allocation**: Material matching with extensive waste inventories
- **Concurrent operations**: Multiple users processing GRNs and production simultaneously
- **Reporting efficiency**: Fast generation of utilization and waste analytics