/**
 * Product Seeder
 * Creates sample products for testing and development
 */

/**
 * Seed products into database
 * @param {Object} db - Database models object
 */
const seedProducts = async (db) => {
  // Import Product and ProductVariant models from database
  const { Product, ProductVariant } = db;
  
  // Check if products already exist
  const existingProducts = await Product.findAll({
    limit: 1, // Just check if any products exist
  });
  
  // If products already exist, skip seeding
  if (existingProducts.length > 0) {
    console.log('⚠️  Products already exist. Skipping product seeding.');
    return; // Exit function
  }
  
  // Sample products to create
  const sampleProducts = [
    {
      name: 'Cotton T-Shirt', // Product name
      sku: 'TSH-001', // Product SKU
      product_type: 'FG', // Finished Goods
      track_inventory: true, // Track inventory
      description: 'High quality cotton t-shirt, comfortable and durable', // Description
      default_vat_rate: 16.0, // Default VAT rate (16%)
      variants: [
        { sku: 'TSH-001-S', attributes: { size: 'S', color: 'Red' } }, // Small Red
        { sku: 'TSH-001-M', attributes: { size: 'M', color: 'Red' } }, // Medium Red
        { sku: 'TSH-001-L', attributes: { size: 'L', color: 'Red' } }, // Large Red
        { sku: 'TSH-001-XL', attributes: { size: 'XL', color: 'Blue' } }, // XL Blue
      ],
    },
    {
      name: 'Jeans', // Product name
      sku: 'JNS-001', // Product SKU
      product_type: 'FG', // Finished Goods
      track_inventory: true, // Track inventory
      description: 'Classic blue denim jeans', // Description
      default_vat_rate: 16.0, // Default VAT rate (16%)
      variants: [
        { sku: 'JNS-001-30', attributes: { waist: '30', length: '32', color: 'Blue' } }, // Waist 30, Length 32
        { sku: 'JNS-001-32', attributes: { waist: '32', length: '32', color: 'Blue' } }, // Waist 32, Length 32
        { sku: 'JNS-001-34', attributes: { waist: '34', length: '34', color: 'Blue' } }, // Waist 34, Length 34
      ],
    },
    {
      name: 'Laptop Computer', // Product name
      sku: 'LAP-001', // Product SKU
      product_type: 'FG', // Finished Goods
      track_inventory: true, // Track inventory
      description: '15.6 inch laptop, Intel i7 processor, 16GB RAM, 512GB SSD', // Description
      default_vat_rate: 16.0, // Default VAT rate (16%)
      variants: [], // No variants for this product
    },
    {
      name: 'Mobile Phone', // Product name
      sku: 'PHN-001', // Product SKU
      product_type: 'FG', // Finished Goods
      track_inventory: true, // Track inventory
      description: 'Smartphone, 128GB storage, dual SIM', // Description
      default_vat_rate: 16.0, // Default VAT rate (16%)
      variants: [
        { sku: 'PHN-001-64', attributes: { storage: '64GB', color: 'Black' } }, // 64GB Black
        { sku: 'PHN-001-128', attributes: { storage: '128GB', color: 'Black' } }, // 128GB Black
        { sku: 'PHN-001-256', attributes: { storage: '256GB', color: 'White' } }, // 256GB White
      ],
    },
    {
      name: 'Cotton Fabric', // Product name (Raw Material)
      sku: 'FAB-001', // Product SKU
      product_type: 'RM', // Raw Material
      track_inventory: true, // Track inventory
      description: 'Cotton fabric, 100% cotton, white color', // Description
      default_vat_rate: 16.0, // Default VAT rate (16%)
      variants: [], // No variants for raw materials typically
    },
    {
      name: 'Electronic Components', // Product name (Raw Material)
      sku: 'ELC-001', // Product SKU
      product_type: 'RM', // Raw Material
      track_inventory: true, // Track inventory
      description: 'Electronic components for manufacturing', // Description
      default_vat_rate: 16.0, // Default VAT rate (16%)
      variants: [], // No variants
    },
  ];
  
  // Create products with variants
  for (const productData of sampleProducts) {
    // Extract variants
    const { variants, ...productFields } = productData;
    
    // Create product
    const product = await Product.create(productFields);
    console.log(`✅ Created product: ${product.name} (${product.sku})`);
    
    // Create variants if provided
    if (variants && variants.length > 0) {
      for (const variantData of variants) {
        // Create variant
        await ProductVariant.create({
          product_id: product.id, // Link to product
          sku: variantData.sku, // Variant SKU
          attributes: variantData.attributes, // Variant attributes
        });
        console.log(`   ✅ Created variant: ${variantData.sku}`);
      }
    }
  }
  
  console.log('✅ Product seeding completed!');
};

// Export seed function
module.exports = seedProducts;

