/**
 * Price List Seeder
 * Creates default price lists (retail, wholesale, B2B)
 */

/**
 * Seed price lists into database
 * @param {Object} db - Database models object
 */
const seedPriceLists = async (db) => {
  // Import PriceList model
  const { PriceList } = db;
  
  // Check if price lists already exist
  const existingPriceLists = await PriceList.findAll();
  
  // If price lists already exist, skip seeding
  if (existingPriceLists.length > 0) {
    console.log('⚠️  Price lists already exist. Skipping price list seeding.');
    return; // Exit function
  }
  
  // Create default retail price list (set as default)
  const retailPriceList = await PriceList.create({
    name: 'Retail', // Price list name
    code: 'RETAIL', // Price list code
    price_list_type: 'RETAIL', // Price list type
    currency: 'KES', // Currency (Kenya Shilling)
    is_default: true, // Set as default price list
    active: true, // Active status
    description: 'Standard retail pricing', // Description
  });
  console.log('✅ Created price list: Retail (default)');
  
  // Create wholesale price list
  const wholesalePriceList = await PriceList.create({
    name: 'Wholesale', // Price list name
    code: 'WHOLESALE', // Price list code
    price_list_type: 'WHOLESALE', // Price list type
    currency: 'KES', // Currency (Kenya Shilling)
    is_default: false, // Not default
    active: true, // Active status
    description: 'Wholesale pricing for bulk purchases', // Description
  });
  console.log('✅ Created price list: Wholesale');
  
  // Create B2B price list
  const b2bPriceList = await PriceList.create({
    name: 'B2B', // Price list name
    code: 'B2B', // Price list code
    price_list_type: 'B2B', // Price list type
    currency: 'KES', // Currency (Kenya Shilling)
    is_default: false, // Not default
    active: true, // Active status
    description: 'Business-to-business pricing', // Description
  });
  console.log('✅ Created price list: B2B');
  
  console.log('✅ Price list seeding completed!');
};

// Export seed function
module.exports = seedPriceLists;

