/**
 * Sales Management Routes
 * Route definitions for sales and POS management endpoints
 */

// Import Express Router
const express = require('express');
// Create router instance
const router = express.Router();
// Import sales controllers
const salesControllers = require('../controllers');
// Import sales validations
const {
  validateCreateSale,
  validateGetSale,
  validateListSales,
  validateCancelSale,
  validateDispatchSale,
  validateCheckStock,
} = require('../validations');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');

/**
 * @swagger
 * components:
 *   schemas:
 *     Sale:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *         invoice_no:
 *           type: string
 *         customer_id:
 *           type: integer
 *           nullable: true
 *         sale_type:
 *           type: string
 *           enum: [POS, INVOICE]
 *         user_id:
 *           type: integer
 *         status:
 *           type: string
 *           enum: [DRAFT, PAID, CANCELLED]
 *         subtotal:
 *           type: number
 *         vat:
 *           type: number
 *         total:
 *           type: number
 */

/**
 * @swagger
 * /sales:
 *   get:
 *     summary: List sales
 *     description: Get a paginated list of sales with optional filters
 *     tags: [Sales]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *         description: Items per page
 *       - in: query
 *         name: customer_id
 *         schema:
 *           type: integer
 *         description: Filter by customer ID
 *       - in: query
 *         name: status
 *         schema:
 *           type: string
 *           enum: [DRAFT, PAID, CANCELLED]
 *         description: Filter by status
 *       - in: query
 *         name: sale_type
 *         schema:
 *           type: string
 *           enum: [POS, INVOICE]
 *         description: Filter by sale type
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date-time
 *         description: Filter by start date (ISO 8601)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date-time
 *         description: Filter by end date (ISO 8601)
 *       - in: query
 *         name: user_id
 *         schema:
 *           type: integer
 *         description: Filter by cashier user ID
 *     responses:
 *       200:
 *         description: Sales retrieved successfully
 *       400:
 *         description: Validation error
 *       401:
 *         description: Unauthorized
 */
router.get('/', authenticate, validateListSales, salesControllers.listSales);

/**
 * @swagger
 * /sales:
 *   post:
 *     summary: Create new sale
 *     description: Create a new POS sale with items. Inventory is reserved on creation.
 *     tags: [Sales]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - items
 *             properties:
 *               customer_id:
 *                 type: integer
 *                 description: Customer ID (optional, for walk-in sales)
 *               sale_type:
 *                 type: string
 *                 enum: [POS, INVOICE]
 *                 default: POS
 *               items:
 *                 type: array
 *                 items:
 *                   type: object
 *                   required:
 *                     - product_id
 *                     - quantity
 *                   properties:
 *                     product_id:
 *                       type: integer
 *                     variant_id:
 *                       type: integer
 *                     quantity:
 *                       type: number
 *                     unit_price:
 *                       type: number
 *     responses:
 *       201:
 *         description: Sale created successfully
 *       400:
 *         description: Validation error or insufficient stock
 *       401:
 *         description: Unauthorized
 */
router.post('/', authenticate, validateCreateSale, salesControllers.createSale);

/**
 * @swagger
 * /sales/check-stock:
 *   post:
 *     summary: Bulk check stock availability
 *     description: Check stock availability for multiple cart items at once
 *     tags: [Sales]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - items
 *             properties:
 *               items:
 *                 type: array
 *                 items:
 *                   type: object
 *                   required:
 *                     - product_id
 *                     - quantity
 *                   properties:
 *                     product_id:
 *                       type: integer
 *                     variant_id:
 *                       type: integer
 *                     quantity:
 *                       type: number
 *     responses:
 *       200:
 *         description: Stock checked successfully
 *       400:
 *         description: Validation error
 *       401:
 *         description: Unauthorized
 */
router.post('/check-stock', authenticate, validateCheckStock, salesControllers.checkStock);

/**
 * @swagger
 * /sales/{id}:
 *   get:
 *     summary: Get sale by ID
 *     description: Get sale details with items and associations
 *     tags: [Sales]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Sale ID
 *     responses:
 *       200:
 *         description: Sale retrieved successfully
 *       400:
 *         description: Validation error
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Sale not found
 */
router.get('/:id', authenticate, validateGetSale, salesControllers.getSale);

/**
 * @swagger
 * /sales/{id}/cancel:
 *   put:
 *     summary: Cancel sale
 *     description: Cancel a sale (only if status is DRAFT). Reserved inventory is released.
 *     tags: [Sales]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Sale ID
 *     responses:
 *       200:
 *         description: Sale cancelled successfully
 *       400:
 *         description: Validation error or sale cannot be cancelled
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Sale not found
 */
router.put('/:id/cancel', authenticate, validateCancelSale, salesControllers.cancelSale);

/**
 * @swagger
 * /sales/{id}/dispatch:
 *   post:
 *     summary: Dispatch sale
 *     description: Dispatch a paid sale - mark items as dispatched and remove from inventory
 *     tags: [Sales]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Sale ID
 *     responses:
 *       200:
 *         description: Sale dispatched successfully
 *       400:
 *         description: Validation error or sale cannot be dispatched
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Sale not found
 */
router.post('/:id/dispatch', authenticate, validateDispatchSale, salesControllers.dispatchSale);

// Export router
module.exports = router;
