/**
 * Returns Routes
 * API routes for return operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import return controllers
const returnControllers = require('../controllers');
// Import validation rules
const returnValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');

/**
 * @swagger
 * /api/returns:
 *   post:
 *     summary: Create return
 *     tags: [Returns]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - sale_id
 *               - items
 *             properties:
 *               sale_id:
 *                 type: integer
 *                 description: Original sale ID
 *               items:
 *                 type: array
 *                 description: Return items
 *                 items:
 *                   type: object
 *                   required:
 *                     - sale_item_id
 *                     - quantity
 *                   properties:
 *                     sale_item_id:
 *                       type: integer
 *                       description: Original sale item ID
 *                     inventory_item_id:
 *                       type: integer
 *                       nullable: true
 *                       description: Inventory item ID (UID) - required for UID-tracked items, omit for quantity-based returns
 *                     quantity:
 *                       type: number
 *                       description: Quantity being returned
 *                     reason:
 *                       type: string
 *                       description: Reason for returning this item
 *               reason:
 *                 type: string
 *                 description: General return reason
 */
router.post(
  '/', // POST /api/returns
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin', 'cashier']), // Require role
  returnValidations.validateCreateReturn, // Validate request
  returnControllers.createReturn // Create return controller
);

/**
 * @swagger
 * /api/returns:
 *   get:
 *     summary: List returns
 *     tags: [Returns]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: sale_id
 *         schema:
 *           type: integer
 *         description: Filter by sale ID
 *       - in: query
 *         name: status
 *         schema:
 *           type: string
 *           enum: [DRAFT, COMPLETED]
 *         description: Filter by status
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *         description: Items per page
 */
router.get(
  '/', // GET /api/returns
  authenticate, // Require authentication
  returnValidations.validateListReturns, // Validate request
  returnControllers.listReturns // List returns controller
);

/**
 * @swagger
 * /api/returns/{id}:
 *   get:
 *     summary: Get return by ID
 *     tags: [Returns]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Return ID
 */
router.get(
  '/:id', // GET /api/returns/:id
  authenticate, // Require authentication
  returnValidations.validateGetReturn, // Validate request
  returnControllers.getReturn // Get return controller
);

/**
 * @swagger
 * /api/returns/{id}/complete:
 *   post:
 *     summary: Complete return (update inventory and generate credit note)
 *     tags: [Returns]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Return ID
 */
router.post(
  '/:id/complete', // POST /api/returns/:id/complete
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  returnValidations.validateCompleteReturn, // Validate request
  returnControllers.completeReturn // Complete return controller
);

// Export router
module.exports = router;
