/**
 * Reports Validation Rules
 * Input validation rules for report endpoints
 */

// Import express-validator for request validation
const { query, validationResult } = require('express-validator');
// Import custom validation error class
const { ValidationError } = require('../../../utils/errors');

/**
 * Validation middleware
 * Handles validation errors and formats response
 */
const handleValidationErrors = (req, res, next) => {
  // Get validation errors
  const errors = validationResult(req); // Get validation errors
  
  // Check if there are errors
  if (!errors.isEmpty()) {
    // Extract error messages
    const errorMessages = errors.array().map(error => ({
      field: error.param, // Field name
      message: error.msg, // Error message
    })); // Map errors to messages
    
    // Throw validation error
    throw new ValidationError('Validation failed', errorMessages); // Throw validation error
  }
  
  // Continue to next middleware if no errors
  next(); // Continue
};

/**
 * Validate date range query parameters
 */
const validateDateRange = [
  // Validate start_date (optional)
  query('start_date')
    .optional()
    .isISO8601()
    .withMessage('Start date must be a valid ISO 8601 date'),
  
  // Validate end_date (optional)
  query('end_date')
    .optional()
    .isISO8601()
    .withMessage('End date must be a valid ISO 8601 date'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate sales report by date request
 */
const validateSalesReportByDate = [
  // Use date range validation
  ...validateDateRange,
];

/**
 * Validate sales report by product request
 */
const validateSalesReportByProduct = [
  // Use date range validation
  ...validateDateRange,
  
  // Validate product_id (optional)
  query('product_id')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Product ID must be a positive integer'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate sales report by customer request
 */
const validateSalesReportByCustomer = [
  // Use date range validation
  ...validateDateRange,
  
  // Validate customer_id (optional)
  query('customer_id')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Customer ID must be a positive integer'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate sales report by cashier request
 */
const validateSalesReportByCashier = [
  // Use date range validation
  ...validateDateRange,
  
  // Validate user_id (optional)
  query('user_id')
    .optional()
    .isInt({ min: 1 })
    .withMessage('User ID must be a positive integer'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate fast-moving products request
 */
const validateFastMovingProducts = [
  // Use date range validation
  ...validateDateRange,
  
  // Validate limit (optional)
  query('limit')
    .optional()
    .isInt({ min: 1, max: 100 })
    .withMessage('Limit must be between 1 and 100'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate slow-moving products request
 */
const validateSlowMovingProducts = [
  // Use date range validation
  ...validateDateRange,
  
  // Validate limit (optional)
  query('limit')
    .optional()
    .isInt({ min: 1, max: 100 })
    .withMessage('Limit must be between 1 and 100'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate low stock items request
 */
const validateLowStockItems = [
  // Validate category_id (optional)
  query('category_id')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Category ID must be a positive integer'),
  
  // Handle validation errors
  handleValidationErrors,
];

/**
 * Validate profit/loss report request
 */
const validateProfitLossReport = [
  // Use date range validation
  ...validateDateRange,
];

/**
 * Validate payment methods report request
 */
const validatePaymentMethodsReport = [
  // Use date range validation
  ...validateDateRange,
  
  // Validate provider (optional)
  query('provider')
    .optional()
    .isIn(['ALL', 'CASH', 'CARD', 'MPESA'])
    .withMessage('Provider must be one of: ALL, CASH, CARD, MPESA'),
  
  // Handle validation errors
  handleValidationErrors,
];

// Tax report validation removed - tax functionality removed from system
// Shift report validation removed - shift functionality removed from system

// Export all validations
module.exports = {
  validateSalesReportByDate,
  validateSalesReportByProduct,
  validateSalesReportByCustomer,
  validateSalesReportByCashier,
  validateFastMovingProducts,
  validateSlowMovingProducts,
  validateLowStockItems,
  validateProfitLossReport,
  validatePaymentMethodsReport,
  // validateTaxReport removed - tax functionality removed
  // validateShiftReport removed - shift functionality removed
};

