/**
 * Reports Routes
 * API routes for report operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import report controllers
const reportControllers = require('../controllers');
// Import validation rules
const reportValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');

// ============================================
// Sales Reports Routes
// ============================================

/**
 * @swagger
 * /api/reports/sales/date:
 *   get:
 *     summary: Get sales report by date range
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 */
router.get(
  '/sales/date', // GET /api/reports/sales/date
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateSalesReportByDate, // Validate request
  reportControllers.getSalesReportByDate // Get sales report by date controller
);

/**
 * @swagger
 * /api/reports/sales/product:
 *   get:
 *     summary: Get sales report by product
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 *       - in: query
 *         name: product_id
 *         schema:
 *           type: integer
 *         description: Filter by product ID
 */
router.get(
  '/sales/product', // GET /api/reports/sales/product
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateSalesReportByProduct, // Validate request
  reportControllers.getSalesReportByProduct // Get sales report by product controller
);

/**
 * @swagger
 * /api/reports/sales/customer:
 *   get:
 *     summary: Get sales report by customer
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 *       - in: query
 *         name: customer_id
 *         schema:
 *           type: integer
 *         description: Filter by customer ID
 */
router.get(
  '/sales/customer', // GET /api/reports/sales/customer
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateSalesReportByCustomer, // Validate request
  reportControllers.getSalesReportByCustomer // Get sales report by customer controller
);

/**
 * @swagger
 * /api/reports/sales/cashier:
 *   get:
 *     summary: Get sales report by cashier
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 *       - in: query
 *         name: user_id
 *         schema:
 *           type: integer
 *         description: Filter by cashier (user) ID
 */
router.get(
  '/sales/cashier', // GET /api/reports/sales/cashier
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateSalesReportByCashier, // Validate request
  reportControllers.getSalesReportByCashier // Get sales report by cashier controller
);

// ============================================
// Inventory Reports Routes
// ============================================

/**
 * @swagger
 * /api/reports/inventory/fast-moving:
 *   get:
 *     summary: Get fast-moving products report
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *         description: "Maximum number of results (default: 20)"
 */
router.get(
  '/inventory/fast-moving', // GET /api/reports/inventory/fast-moving
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateFastMovingProducts, // Validate request
  reportControllers.getFastMovingProducts // Get fast-moving products controller
);

/**
 * @swagger
 * /api/reports/inventory/slow-moving:
 *   get:
 *     summary: Get slow-moving products report
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *         description: "Maximum number of results (default: 20)"
 */
router.get(
  '/inventory/slow-moving', // GET /api/reports/inventory/slow-moving
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateSlowMovingProducts, // Validate request
  reportControllers.getSlowMovingProducts // Get slow-moving products controller
);

/**
 * @swagger
 * /api/reports/inventory/low-stock:
 *   get:
 *     summary: Get low stock items report (reorder alerts)
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: category_id
 *         schema:
 *           type: integer
 *         description: Filter by category ID
 */
router.get(
  '/inventory/low-stock', // GET /api/reports/inventory/low-stock
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateLowStockItems, // Validate request
  reportControllers.getLowStockItems // Get low stock items controller
);

// ============================================
// Financial Reports Routes
// ============================================

/**
 * @swagger
 * /api/reports/financial/profit-loss:
 *   get:
 *     summary: Get profit/loss report
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 */
router.get(
  '/financial/profit-loss', // GET /api/reports/financial/profit-loss
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validateProfitLossReport, // Validate request
  reportControllers.getProfitLossReport // Get profit/loss report controller
);

// ============================================
// Payment Methods Reports Routes
// ============================================

/**
 * @swagger
 * /api/reports/payment-methods:
 *   get:
 *     summary: Get payment methods report (CASH, CARD, MPESA)
 *     tags: [Reports]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: start_date
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date (ISO 8601 format)
 *       - in: query
 *         name: end_date
 *         schema:
 *           type: string
 *           format: date
 *         description: End date (ISO 8601 format)
 *       - in: query
 *         name: provider
 *         schema:
 *           type: string
 *           enum: [ALL, CASH, CARD, MPESA]
 *         description: Filter by payment provider (optional)
 */
router.get(
  '/payment-methods', // GET /api/reports/payment-methods
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  reportValidations.validatePaymentMethodsReport, // Validate request
  reportControllers.getPaymentMethodsReport // Get payment methods report controller
);

// ============================================
// Tax Reports Routes - REMOVED (tax functionality removed)
// ============================================

// ============================================
// Shift Reports Routes - REMOVED (shift functionality removed)
// ============================================

// Export router
module.exports = router;

