/**
 * Product Management Routes
 * Route definitions for product management endpoints
 */

// Import Express Router
const express = require('express');
// Create router instance
const router = express.Router();
// Import product controllers
const productControllers = require('../controllers');
// Import product validations
const {
  validateCreateProduct,
  validateUpdateProduct,
  validateGetProduct,
  validateDeleteProduct,
  validateListProducts,
  validateSearchProducts,
} = require('../validations');
// Import multipart form data validations
const {
  validateCreateProductMultipart,
  validateUpdateProductMultipart,
} = require('../validations/multipart');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');
// Import upload middleware
const { uploadProductImage, handleUploadError } = require('../../../middlewares/upload');

/**
 * @swagger
 * /products:
 *   get:
 *     summary: List products
 *     description: Get a paginated list of products with optional filters
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 10
 *         description: Number of products per page
 *       - in: query
 *         name: search
 *         schema:
 *           type: string
 *         description: Search term (searches in name and SKU)
 *       - in: query
 *         name: product_type
 *         schema:
 *           type: string
 *           enum: [FG, RM]
 *         description: Filter by product type
 *       - in: query
 *         name: track_inventory
 *         schema:
 *           type: string
 *           enum: [true, false]
 *         description: Filter by inventory tracking
 *     responses:
 *       200:
 *         description: Products retrieved successfully
 *       401:
 *         description: Unauthorized
 */
router.get('/', authenticate, validateListProducts, productControllers.listProducts);

/**
 * @swagger
 * /products/search:
 *   get:
 *     summary: Search products
 *     description: Search products by name or SKU
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: q
 *         schema:
 *           type: string
 *         required: true
 *         description: Search term
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 20
 *         description: Maximum number of results
 *     responses:
 *       200:
 *         description: Search completed successfully
 *       401:
 *         description: Unauthorized
 */
router.get('/search', authenticate, validateSearchProducts, productControllers.searchProducts);

/**
 * @swagger
 * /products/{id}:
 *   get:
 *     summary: Get product by ID
 *     description: Retrieve a specific product by ID
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     responses:
 *       200:
 *         description: Product retrieved successfully
 *       404:
 *         description: Product not found
 *       401:
 *         description: Unauthorized
 */
router.get('/:id', authenticate, validateGetProduct, productControllers.getProductById);

/**
 * @swagger
 * /products/{id}/availability:
 *   get:
 *     summary: Get product stock availability
 *     description: Get real-time stock availability for a product
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     responses:
 *       200:
 *         description: Availability retrieved successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Product not found
 */
router.get('/:id/availability', authenticate, validateGetProduct, productControllers.getProductAvailability);

/**
 * @swagger
 * /products:
 *   post:
 *     summary: Create new product
 *     description: Create a new product with optional image
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             required:
 *               - name
 *               - product_type
 *             properties:
 *               name:
 *                 type: string
 *                 example: T-Shirt
 *               sku:
 *                 type: string
 *                 example: TSH-001
 *               product_type:
 *                 type: string
 *                 enum: [FG, RM]
 *                 example: FG
 *               track_inventory:
 *                 type: boolean
 *                 default: true
 *               description:
 *                 type: string
 *                 example: High quality cotton t-shirt
 *               low_stock_threshold:
 *                 type: number
 *                 example: 10
 *               image:
 *                 type: string
 *                 format: binary
 *                 description: Product image file (optional)
 *     responses:
 *       201:
 *         description: Product created successfully
 *       400:
 *         description: Validation error
 *       401:
 *         description: Unauthorized
 */
router.post('/', authenticate, requireRole(['manager', 'system_admin']), uploadProductImage, handleUploadError, validateCreateProductMultipart, productControllers.createProduct);

/**
 * @swagger
 * /products/{id}:
 *   put:
 *     summary: Update product
 *     description: Update product information with optional image
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               sku:
 *                 type: string
 *               product_type:
 *                 type: string
 *                 enum: [FG, RM]
 *               track_inventory:
 *                 type: boolean
 *               description:
 *                 type: string
 *               low_stock_threshold:
 *                 type: number
 *               image:
 *                 type: string
 *                 format: binary
 *                 description: Product image file (optional)
 *     responses:
 *       200:
 *         description: Product updated successfully
 *       400:
 *         description: Validation error
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Product not found
 */
router.put('/:id', authenticate, requireRole(['manager', 'system_admin']), uploadProductImage, handleUploadError, validateUpdateProductMultipart, productControllers.updateProduct);

/**
 * @swagger
 * /products/{id}/image:
 *   post:
 *     summary: Upload product image
 *     description: Upload an image for a specific product
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             required:
 *               - image
 *             properties:
 *               image:
 *                 type: string
 *                 format: binary
 *                 description: Product image file
 *     responses:
 *       200:
 *         description: Image uploaded successfully
 *       400:
 *         description: No image file provided or validation error
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Product not found
 */
router.post('/:id/image', authenticate, requireRole(['manager', 'system_admin']), validateGetProduct, uploadProductImage, handleUploadError, productControllers.uploadProductImage);

/**
 * @swagger
 * /products/{id}:
 *   delete:
 *     summary: Delete product
 *     description: Delete a product (not allowed - products are kept for audit purposes)
 *     tags: [Products]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     responses:
 *       400:
 *         description: Product deletion not allowed
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Product not found
 */
router.delete('/:id', authenticate, requireRole(['manager', 'system_admin']), validateDeleteProduct, productControllers.deleteProduct);

// Export router
module.exports = router;
