/**
 * Procurement Routes
 * API routes for procurement operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import procurement controllers
const procurementControllers = require('../controllers');
// Import validation rules
const procurementValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');

// ============================================
// Vendor Routes
// ============================================

/**
 * @swagger
 * /api/procurement/vendors:
 *   post:
 *     summary: Create vendor
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.post(
  '/vendors', // POST /api/procurement/vendors
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateCreateVendor, // Validate request
  procurementControllers.createVendor // Create vendor controller
);

/**
 * @swagger
 * /api/procurement/vendors:
 *   get:
 *     summary: List vendors
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.get(
  '/vendors', // GET /api/procurement/vendors
  authenticate, // Require authentication
  procurementValidations.validateListVendors, // Validate request
  procurementControllers.listVendors // List vendors controller
);

/**
 * @swagger
 * /api/procurement/vendors/{id}:
 *   get:
 *     summary: Get vendor by ID
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.get(
  '/vendors/:id', // GET /api/procurement/vendors/:id
  authenticate, // Require authentication
  procurementValidations.validateGetVendor, // Validate request
  procurementControllers.getVendor // Get vendor controller
);

/**
 * @swagger
 * /api/procurement/vendors/{id}:
 *   put:
 *     summary: Update vendor
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.put(
  '/vendors/:id', // PUT /api/procurement/vendors/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateUpdateVendor, // Validate request
  procurementControllers.updateVendor // Update vendor controller
);

/**
 * @swagger
 * /api/procurement/vendors/{id}:
 *   delete:
 *     summary: Delete vendor
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.delete(
  '/vendors/:id', // DELETE /api/procurement/vendors/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateDeleteVendor, // Validate request
  procurementControllers.deleteVendor // Delete vendor controller
);

// ============================================
// Purchase Order Routes
// ============================================

/**
 * @swagger
 * /api/procurement/purchase-orders:
 *   post:
 *     summary: Create purchase order
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.post(
  '/purchase-orders', // POST /api/procurement/purchase-orders
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateCreatePurchaseOrder, // Validate request
  procurementControllers.createPurchaseOrder // Create purchase order controller
);

/**
 * @swagger
 * /api/procurement/purchase-orders:
 *   get:
 *     summary: List purchase orders
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.get(
  '/purchase-orders', // GET /api/procurement/purchase-orders
  authenticate, // Require authentication
  procurementValidations.validateListPurchaseOrders, // Validate request
  procurementControllers.listPurchaseOrders // List purchase orders controller
);

/**
 * @swagger
 * /api/procurement/purchase-orders/{id}:
 *   get:
 *     summary: Get purchase order by ID
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.get(
  '/purchase-orders/:id', // GET /api/procurement/purchase-orders/:id
  authenticate, // Require authentication
  procurementValidations.validateGetPurchaseOrder, // Validate request
  procurementControllers.getPurchaseOrder // Get purchase order controller
);

/**
 * @swagger
 * /api/procurement/purchase-orders/{id}:
 *   put:
 *     summary: Update purchase order
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.put(
  '/purchase-orders/:id', // PUT /api/procurement/purchase-orders/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateUpdatePurchaseOrder, // Validate request
  procurementControllers.updatePurchaseOrder // Update purchase order controller
);

/**
 * @swagger
 * /api/procurement/purchase-orders/{id}:
 *   delete:
 *     summary: Delete purchase order
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.delete(
  '/purchase-orders/:id', // DELETE /api/procurement/purchase-orders/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateDeletePurchaseOrder, // Validate request
  procurementControllers.deletePurchaseOrder // Delete purchase order controller
);

/**
 * @swagger
 * /api/procurement/purchase-orders/{id}/status:
 *   put:
 *     summary: Update purchase order status
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.put(
  '/purchase-orders/:id/status', // PUT /api/procurement/purchase-orders/:id/status
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateUpdatePurchaseOrderStatus, // Validate request
  procurementControllers.updatePurchaseOrderStatus // Update purchase order status controller
);

// ============================================
// GRN Routes
// ============================================

/**
 * @swagger
 * /api/procurement/grns:
 *   post:
 *     summary: Create GRN
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.post(
  '/grns', // POST /api/procurement/grns
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateCreateGRN, // Validate request
  procurementControllers.createGRN // Create GRN controller
);

/**
 * @swagger
 * /api/procurement/grns:
 *   get:
 *     summary: List GRNs
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.get(
  '/grns', // GET /api/procurement/grns
  authenticate, // Require authentication
  procurementValidations.validateListGRNs, // Validate request
  procurementControllers.listGRNs // List GRNs controller
);

/**
 * @swagger
 * /api/procurement/grns/{id}:
 *   get:
 *     summary: Get GRN by ID
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.get(
  '/grns/:id', // GET /api/procurement/grns/:id
  authenticate, // Require authentication
  procurementValidations.validateGetGRN, // Validate request
  procurementControllers.getGRN // Get GRN controller
);

/**
 * @swagger
 * /api/procurement/grns/{id}/process:
 *   post:
 *     summary: Process GRN (add stock to inventory)
 *     tags: [Procurement]
 *     security:
 *       - bearerAuth: []
 */
router.post(
  '/grns/:id/process', // POST /api/procurement/grns/:id/process
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  procurementValidations.validateProcessGRN, // Validate request
  procurementControllers.processGRN // Process GRN controller
);

// Export router
module.exports = router;
