/**
 * Pricing Routes
 * API routes for pricing operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import pricing controllers
const pricingControllers = require('../controllers');
// Import validation rules
const pricingValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole, requireAnyRole } = require('../../../middlewares/auth');

// ============================================
// Price List Routes
// ============================================

/**
 * @swagger
 * /api/pricing/price-lists:
 *   post:
 *     summary: Create price list
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - name
 *               - code
 *               - price_list_type
 *             properties:
 *               name:
 *                 type: string
 *                 description: Price list name
 *               code:
 *                 type: string
 *                 description: Price list code (unique)
 *               price_list_type:
 *                 type: string
 *                 enum: [RETAIL, WHOLESALE, B2B, CUSTOMER_SPECIFIC]
 *                 description: Price list type
 *               currency:
 *                 type: string
 *                 description: "Currency code (default: KES)"
 *               description:
 *                 type: string
 *                 description: Price list description
 *               is_default:
 *                 type: boolean
 *                 description: Whether this is the default price list
 *     responses:
 *       201:
 *         description: Price list created successfully
 *       400:
 *         description: Validation error
 */
router.post(
  '/price-lists', // POST /api/pricing/price-lists
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  pricingValidations.validateCreatePriceList, // Validate request
  pricingControllers.createPriceList // Create price list controller
);

/**
 * @swagger
 * /api/pricing/price-lists:
 *   get:
 *     summary: List price lists
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 10
 *         description: Items per page
 *       - in: query
 *         name: price_list_type
 *         schema:
 *           type: string
 *           enum: [RETAIL, WHOLESALE, B2B, CUSTOMER_SPECIFIC]
 *         description: Filter by price list type
 *       - in: query
 *         name: active
 *         schema:
 *           type: boolean
 *         description: Filter by active status
 *     responses:
 *       200:
 *         description: Price lists retrieved successfully
 */
router.get(
  '/price-lists', // GET /api/pricing/price-lists
  authenticate, // Require authentication
  pricingValidations.validateListPriceLists, // Validate request
  pricingControllers.listPriceLists // List price lists controller
);

/**
 * @swagger
 * /api/pricing/price-lists/default:
 *   get:
 *     summary: Get default price list
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: Default price list retrieved successfully
 */
router.get(
  '/price-lists/default', // GET /api/pricing/price-lists/default
  authenticate, // Require authentication
  pricingControllers.getDefaultPriceList // Get default price list controller
);

/**
 * @swagger
 * /api/pricing/price-lists/{id}:
 *   get:
 *     summary: Get price list by ID
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Price list ID
 *     responses:
 *       200:
 *         description: Price list retrieved successfully
 *       404:
 *         description: Price list not found
 */
router.get(
  '/price-lists/:id', // GET /api/pricing/price-lists/:id
  authenticate, // Require authentication
  pricingValidations.validateGetPriceList, // Validate request
  pricingControllers.getPriceList // Get price list controller
);

/**
 * @swagger
 * /api/pricing/price-lists/{id}:
 *   put:
 *     summary: Update price list
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Price list ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               code:
 *                 type: string
 *               price_list_type:
 *                 type: string
 *                 enum: [RETAIL, WHOLESALE, B2B, CUSTOMER_SPECIFIC]
 *               currency:
 *                 type: string
 *               description:
 *                 type: string
 *               is_default:
 *                 type: boolean
 *               active:
 *                 type: boolean
 *     responses:
 *       200:
 *         description: Price list updated successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Price list not found
 */
router.put(
  '/price-lists/:id', // PUT /api/pricing/price-lists/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  pricingValidations.validateUpdatePriceList, // Validate request
  pricingControllers.updatePriceList // Update price list controller
);

/**
 * @swagger
 * /api/pricing/price-lists/{id}:
 *   delete:
 *     summary: Delete price list
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Price list ID
 *     responses:
 *       200:
 *         description: Price list deleted successfully
 *       400:
 *         description: Validation error (has product prices or customers)
 *       404:
 *         description: Price list not found
 */
router.delete(
  '/price-lists/:id', // DELETE /api/pricing/price-lists/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  pricingValidations.validateDeletePriceList, // Validate request
  pricingControllers.deletePriceList // Delete price list controller
);

// ============================================
// Product Price Routes
// ============================================

/**
 * @swagger
 * /api/pricing/prices:
 *   post:
 *     summary: Create product price
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - price_list_id
 *               - product_id
 *               - price
 *             properties:
 *               price_list_id:
 *                 type: integer
 *                 description: Price list ID
 *               product_id:
 *                 type: integer
 *                 description: Product ID
 *               price:
 *                 type: number
 *                 description: Unit price
 *               min_quantity:
 *                 type: number
 *                 description: "Minimum quantity for this price (default: 1)"
 *               max_quantity:
 *                 type: number
 *                 nullable: true
 *                 description: Maximum quantity for this price (null for no limit)
 *               effective_from:
 *                 type: string
 *                 format: date-time
 *                 description: "Effective start date (default: now)"
 *               effective_to:
 *                 type: string
 *                 format: date-time
 *                 nullable: true
 *                 description: Effective end date (null for ongoing)
 *               active:
 *                 type: boolean
 *                 description: "Whether this price is active (default: true)"
 *     responses:
 *       201:
 *         description: Product price created successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Price list or product not found
 */
router.post(
  '/prices', // POST /api/pricing/prices
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  pricingValidations.validateCreateProductPrice, // Validate request
  pricingControllers.createProductPrice // Create product price controller
);

/**
 * @swagger
 * /api/pricing/prices:
 *   get:
 *     summary: List product prices
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 10
 *         description: Items per page
 *       - in: query
 *         name: price_list_id
 *         schema:
 *           type: integer
 *         description: Filter by price list ID
 *       - in: query
 *         name: product_id
 *         schema:
 *           type: integer
 *         description: Filter by product ID
 *       - in: query
 *         name: active
 *         schema:
 *           type: boolean
 *         description: Filter by active status
 *     responses:
 *       200:
 *         description: Product prices retrieved successfully
 */
router.get(
  '/prices', // GET /api/pricing/prices
  authenticate, // Require authentication
  pricingValidations.validateListProductPrices, // Validate request
  pricingControllers.listProductPrices // List product prices controller
);

/**
 * @swagger
 * /api/pricing/prices/lookup:
 *   get:
 *     summary: Get price for product
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: product_id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *       - in: query
 *         name: quantity
 *         schema:
 *           type: number
 *           default: 1
 *         description: Quantity (for quantity-based pricing)
 *       - in: query
 *         name: price_list_id
 *         schema:
 *           type: integer
 *         description: Price list ID (optional, uses default if not provided)
 *     responses:
 *       200:
 *         description: Price retrieved successfully
 *       404:
 *         description: Price not found
 */
router.get(
  '/prices/lookup', // GET /api/pricing/prices/lookup
  authenticate, // Require authentication
  pricingValidations.validateGetPrice, // Validate request
  pricingControllers.getPrice // Get price controller
);

/**
 * @swagger
 * /api/pricing/prices/customer:
 *   get:
 *     summary: Get price for customer (walk-in customers supported, customerId = null)
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: product_id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *       - in: query
 *         name: quantity
 *         schema:
 *           type: number
 *           default: 1
 *         description: Quantity (for quantity-based pricing)
 *       - in: query
 *         name: customer_id
 *         schema:
 *           type: integer
 *         description: Customer ID (optional, uses customer's price list if provided)
 *     responses:
 *       200:
 *         description: Price for customer retrieved successfully
 *       404:
 *         description: Price not found
 */
router.get(
  '/prices/customer', // GET /api/pricing/prices/customer
  authenticate, // Require authentication
  pricingValidations.validateGetPriceForCustomer, // Validate request
  pricingControllers.getPriceForCustomer // Get price for customer controller
);

/**
 * @swagger
 * /api/pricing/prices/{id}:
 *   get:
 *     summary: Get product price by ID
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product price ID
 *     responses:
 *       200:
 *         description: Product price retrieved successfully
 *       404:
 *         description: Product price not found
 */
router.get(
  '/prices/:id', // GET /api/pricing/prices/:id
  authenticate, // Require authentication
  pricingValidations.validateGetProductPrice, // Validate request
  pricingControllers.getProductPrice // Get product price controller
);

/**
 * @swagger
 * /api/pricing/prices/{id}:
 *   put:
 *     summary: Update product price
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product price ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               price:
 *                 type: number
 *               min_quantity:
 *                 type: number
 *               max_quantity:
 *                 type: number
 *                 nullable: true
 *               effective_from:
 *                 type: string
 *                 format: date-time
 *               effective_to:
 *                 type: string
 *                 format: date-time
 *                 nullable: true
 *               active:
 *                 type: boolean
 *     responses:
 *       200:
 *         description: Product price updated successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Product price not found
 */
router.put(
  '/prices/:id', // PUT /api/pricing/prices/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  pricingValidations.validateUpdateProductPrice, // Validate request
  pricingControllers.updateProductPrice // Update product price controller
);

/**
 * @swagger
 * /api/pricing/prices/{id}:
 *   delete:
 *     summary: Delete product price
 *     tags: [Pricing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product price ID
 *     responses:
 *       200:
 *         description: Product price deleted successfully
 *       404:
 *         description: Product price not found
 */
router.delete(
  '/prices/:id', // DELETE /api/pricing/prices/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  pricingValidations.validateDeleteProductPrice, // Validate request
  pricingControllers.deleteProductPrice // Delete product price controller
);

// Export router
module.exports = router;

