/**
 * Label Printing Validations
 * Validation rules for label printing API endpoints
 */

// Import express-validator
const { body, query, param, validationResult } = require('express-validator');

/**
 * Validation middleware
 * Checks validation results and returns errors if any
 */
const validate = (req, res, next) => {
  // Get validation errors
  const errors = validationResult(req); // Get validation errors
  
  // If there are errors, return 400 with error details
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false, // Request failed
      message: 'Validation error', // Error message
      errors: errors.array(), // Validation errors
    });
  }
  
  // Continue to next middleware
  next(); // Proceed
};

/**
 * Validation rules for printing inventory item label
 */
const validatePrintInventoryItemLabel = [
  // Validate id parameter
  param('id')
    .notEmpty()
    .withMessage('Inventory item ID is required')
    .isInt({ min: 1 })
    .withMessage('Inventory item ID must be a positive integer')
    .toInt(),
  
  // Validate printer configuration (optional)
  body('printer.method')
    .optional()
    .isIn(['network', 'usb'])
    .withMessage('Printer method must be "network" or "usb"'),
  
  body('printer.host')
    .optional()
    .isString()
    .withMessage('Printer host must be a string')
    .isLength({ min: 1 })
    .withMessage('Printer host cannot be empty'),
  
  body('printer.port')
    .optional()
    .isInt({ min: 1, max: 65535 })
    .withMessage('Printer port must be between 1 and 65535')
    .toInt(),
  
  body('printer.devicePath')
    .optional()
    .isString()
    .withMessage('Printer device path must be a string')
    .isLength({ min: 1 })
    .withMessage('Printer device path cannot be empty'),
  
  // Run validation
  validate,
];

/**
 * Validation rules for printing product label
 */
const validatePrintProductLabel = [
  // Validate id parameter
  param('id')
    .notEmpty()
    .withMessage('Product ID is required')
    .isInt({ min: 1 })
    .withMessage('Product ID must be a positive integer')
    .toInt(),
  
  // Validate printer configuration (optional)
  body('printer.method')
    .optional()
    .isIn(['network', 'usb'])
    .withMessage('Printer method must be "network" or "usb"'),
  
  body('printer.host')
    .optional()
    .isString()
    .withMessage('Printer host must be a string'),
  
  body('printer.port')
    .optional()
    .isInt({ min: 1, max: 65535 })
    .withMessage('Printer port must be between 1 and 65535')
    .toInt(),
  
  body('printer.devicePath')
    .optional()
    .isString()
    .withMessage('Printer device path must be a string'),
  
  // Validate sample UID (optional)
  body('sampleUID')
    .optional()
    .isString()
    .withMessage('Sample UID must be a string')
    .isLength({ min: 1, max: 100 })
    .withMessage('Sample UID must be between 1 and 100 characters'),
  
  // Run validation
  validate,
];

/**
 * Validation rules for generating label preview
 */
const validateGenerateLabelPreview = [
  // Validate inventory_item_id (optional, but at least one of inventory_item_id or product_id required)
  query('inventory_item_id')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Inventory item ID must be a positive integer')
    .toInt(),
  
  // Validate product_id (optional)
  query('product_id')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Product ID must be a positive integer')
    .toInt(),
  
  // Validate uid (optional)
  query('uid')
    .optional()
    .isString()
    .withMessage('UID must be a string')
    .isLength({ min: 1, max: 100 })
    .withMessage('UID must be between 1 and 100 characters'),
  
  // Custom validation: at least one of inventory_item_id or product_id must be provided
  query('inventory_item_id')
    .custom((value, { req }) => {
      if (!value && !req.query.product_id) {
        throw new Error('Either inventory_item_id or product_id is required');
      }
      return true;
    }),
  
  // Run validation
  validate,
];

/**
 * Validation rules for generating ZPL template
 */
const validateGenerateZPLTemplate = [
  // Validate productName
  body('productName')
    .notEmpty()
    .withMessage('Product name is required')
    .isString()
    .withMessage('Product name must be a string')
    .isLength({ min: 1, max: 200 })
    .withMessage('Product name must be between 1 and 200 characters'),
  
  // Validate uid
  body('uid')
    .notEmpty()
    .withMessage('UID is required')
    .isString()
    .withMessage('UID must be a string')
    .isLength({ min: 1, max: 100 })
    .withMessage('UID must be between 1 and 100 characters'),
  
  // Validate qrData (optional)
  body('qrData')
    .optional()
    .isString()
    .withMessage('QR data must be a string')
    .isLength({ min: 1, max: 500 })
    .withMessage('QR data must be between 1 and 500 characters'),
  
  // Run validation
  validate,
];

// Export validation rules
module.exports = {
  validatePrintInventoryItemLabel, // Print inventory item label validation
  validatePrintProductLabel, // Print product label validation
  validateGenerateLabelPreview, // Generate label preview validation
  validateGenerateZPLTemplate, // Generate ZPL template validation
};
