/**
 * Label Printing Routes
 * API routes for label printing operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import label printing controllers
const labelPrintingControllers = require('../controllers');
// Import validation rules
const labelPrintingValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole, requireAnyRole } = require('../../../middlewares/auth');

/**
 * @swagger
 * /api/label-printing/inventory-items/{id}/print:
 *   post:
 *     summary: Print label for inventory item
 *     tags: [Label Printing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Inventory item ID
 *     requestBody:
 *       required: false
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               printer:
 *                 type: object
 *                 properties:
 *                   method:
 *                     type: string
 *                     enum: [network, usb]
 *                     description: Printer method (network or usb)
 *                   host:
 *                     type: string
 *                     description: Printer host (for network printing)
 *                   port:
 *                     type: integer
 *                     default: 9100
 *                     description: Printer port (for network printing)
 *                   devicePath:
 *                     type: string
 *                     description: Printer device path (for USB printing, e.g., /dev/usb/lp0)
 *     responses:
 *       200:
 *         description: Label printed successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Inventory item not found
 */
router.post(
  '/inventory-items/:id/print',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingValidations.validatePrintInventoryItemLabel,
  labelPrintingControllers.printInventoryItemLabel
);

/**
 * @swagger
 * /api/label-printing/products/{id}/print:
 *   post:
 *     summary: Print label for product (generates template/example)
 *     tags: [Label Printing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     requestBody:
 *       required: false
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               printer:
 *                 type: object
 *                 description: Printer configuration (optional, for actual printing)
 *               sampleUID:
 *                 type: string
 *                 description: Sample UID to use (optional)
 *     responses:
 *       200:
 *         description: Label template generated/printed successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Product not found
 */
router.post(
  '/products/:id/print',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingValidations.validatePrintProductLabel,
  labelPrintingControllers.printProductLabel
);

/**
 * @swagger
 * /api/label-printing/preview:
 *   get:
 *     summary: Generate label preview (QR code image)
 *     tags: [Label Printing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: inventory_item_id
 *         schema:
 *           type: integer
 *         description: Inventory item ID (optional, if product_id not provided)
 *       - in: query
 *         name: product_id
 *         schema:
 *           type: integer
 *         description: Product ID (optional, if inventory_item_id not provided)
 *       - in: query
 *         name: uid
 *         schema:
 *           type: string
 *         description: UID to use (optional)
 *     responses:
 *       200:
 *         description: Label preview generated successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Inventory item or product not found
 */
router.get(
  '/preview',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingValidations.validateGenerateLabelPreview,
  labelPrintingControllers.generateLabelPreview
);

/**
 * @swagger
 * /api/label-printing/zpl:
 *   post:
 *     summary: Generate ZPL template (without printing)
 *     tags: [Label Printing]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - productName
 *               - uid
 *             properties:
 *               productName:
 *                 type: string
 *                 description: Product name
 *               uid:
 *                 type: string
 *                 description: UID for the label
 *               qrData:
 *                 type: string
 *                 description: Data to encode in QR code (defaults to uid)
 *     responses:
 *       200:
 *         description: ZPL template generated successfully
 *       400:
 *         description: Validation error
 */
/**
 * @swagger
 * /api/label-printing/inventory-items/{id}/zpl:
 *   get:
 *     summary: Get ZPL data for inventory item (client-side printing)
 *     tags: [Label Printing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Inventory item ID
 *     responses:
 *       200:
 *         description: ZPL data generated successfully
 *       404:
 *         description: Inventory item not found
 */
router.get(
  '/inventory-items/:id/zpl',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingControllers.getInventoryItemLabelZPL
);

/**
 * @swagger
 * /api/label-printing/products/{id}/zpl:
 *   get:
 *     summary: Get ZPL data for product (client-side printing)
 *     tags: [Label Printing]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *       - in: query
 *         name: uid
 *         schema:
 *           type: string
 *         description: Sample UID to use (optional)
 *     responses:
 *       200:
 *         description: ZPL data generated successfully
 *       404:
 *         description: Product not found
 */
router.get(
  '/products/:id/zpl',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingControllers.getProductLabelZPL
);

router.post(
  '/zpl',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingValidations.validateGenerateZPLTemplate,
  labelPrintingControllers.generateZPLTemplate
);

/**
 * Batch ZPL generation for inventory items (client-side/local printing)
 * Accepts an array of inventory_item_ids and returns ZPL data per item.
 */
router.post(
  '/inventory-items/batch-zpl',
  authenticate,
  requireAnyRole(['manager', 'cashier', 'system_admin']),
  labelPrintingControllers.generateBatchInventoryItemZPL
);

// Export router
module.exports = router;
