/**
 * RM Inventory Validation Rules
 * Validation middleware for dimension-based RM inventory endpoints
 */

// Import validation library
const { body, param, query, validationResult } = require('express-validator');

/**
 * Validation middleware to check for validation errors
 */
const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false,
      error: 'Validation failed',
      details: errors.array()
    });
  }
  next();
};

/**
 * Validate list RM inventory pieces request
 */
const validateListRMInventoryPieces = [
  param('productId')
    .isInt({ min: 1 })
    .withMessage('Product ID must be a positive integer'),
  
  query('status')
    .optional()
    .isIn(['FULL', 'USABLE', 'WASTE', 'SCRAP'])
    .withMessage('Status must be one of: FULL, USABLE, WASTE, SCRAP'),
  
  query('page')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Page must be a positive integer'),
  
  query('limit')
    .optional()
    .isInt({ min: 1, max: 100 })
    .withMessage('Limit must be between 1 and 100'),
  
  query('min_length')
    .optional()
    .isFloat({ min: 0 })
    .withMessage('Minimum length must be a non-negative number'),
  
  query('max_length')
    .optional()
    .isFloat({ min: 0 })
    .withMessage('Maximum length must be a non-negative number'),
  
  query('min_width')
    .optional()
    .isFloat({ min: 0 })
    .withMessage('Minimum width must be a non-negative number'),
  
  query('max_width')
    .optional()
    .isFloat({ min: 0 })
    .withMessage('Maximum width must be a non-negative number'),
  
  query('unit')
    .optional()
    .isIn(['inch', 'cm', 'm'])
    .withMessage('Unit must be one of: inch, cm, m'),
  
  handleValidationErrors
];

/**
 * Validate get RM inventory summary request
 */
const validateGetRMInventorySummary = [
  param('productId')
    .isInt({ min: 1 })
    .withMessage('Product ID must be a positive integer'),
  
  query('unit')
    .optional()
    .isIn(['inch', 'cm', 'm'])
    .withMessage('Unit must be one of: inch, cm, m'),
  
  handleValidationErrors
];

/**
 * Validate write off scrap dimensions request
 */
const validateWriteOffScrapDimensions = [
  param('productId')
    .isInt({ min: 1 })
    .withMessage('Product ID must be a positive integer'),
  
  body('piece_ids')
    .isArray({ min: 1 })
    .withMessage('piece_ids must be a non-empty array')
    .custom((value) => {
      if (!value.every(id => Number.isInteger(parseInt(id)) && parseInt(id) > 0)) {
        throw new Error('All piece IDs must be positive integers');
      }
      return true;
    }),
  
  body('reason')
    .notEmpty()
    .withMessage('Reason is required')
    .isLength({ min: 3, max: 255 })
    .withMessage('Reason must be between 3 and 255 characters'),
  
  body('notes')
    .optional()
    .isLength({ max: 1000 })
    .withMessage('Notes must not exceed 1000 characters'),
  
  handleValidationErrors
];

module.exports = {
  validateListRMInventoryPieces,
  validateGetRMInventorySummary,
  validateWriteOffScrapDimensions
};