/**
 * Discount Routes
 * API routes for discount operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import discount controllers
const discountControllers = require('../controllers');
// Import validation rules
const discountValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');

// ============================================
// Discount Routes
// ============================================

/**
 * @swagger
 * /api/discounts:
 *   post:
 *     summary: Create discount
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - name
 *               - discount_type
 *               - discount_value
 *             properties:
 *               name:
 *                 type: string
 *                 description: Discount name
 *               code:
 *                 type: string
 *                 description: Discount code (optional, unique)
 *               discount_type:
 *                 type: string
 *                 enum: [PERCENTAGE, FIXED_AMOUNT, BUY_X_GET_Y]
 *                 description: Discount type
 *               discount_value:
 *                 type: number
 *                 description: Discount value (percentage or amount)
 *               min_purchase_amount:
 *                 type: number
 *                 description: Minimum purchase amount
 *               max_discount_amount:
 *                 type: number
 *                 description: Maximum discount amount
 *               effective_from:
 *                 type: string
 *                 format: date-time
 *                 description: Effective from date
 *               effective_to:
 *                 type: string
 *                 format: date-time
 *                 description: Effective to date
 *               max_uses_per_customer:
 *                 type: integer
 *                 description: Maximum uses per customer
 *               max_total_uses:
 *                 type: integer
 *                 description: Maximum total uses
 *               active:
 *                 type: boolean
 *                 description: Whether discount is active
 *               description:
 *                 type: string
 *                 description: Discount description
 *               rules:
 *                 type: array
 *                 description: Discount rules (optional)
 *                 items:
 *                   type: object
 *                   properties:
 *                     rule_type:
 *                       type: string
 *                       enum: [PRODUCT, CATEGORY, CUSTOMER, ALL]
 *                     product_id:
 *                       type: integer
 *                     category_id:
 *                       type: integer
 *                     customer_id:
 *                       type: integer
 *                     min_quantity:
 *                       type: number
 *     responses:
 *       201:
 *         description: Discount created successfully
 *       400:
 *         description: Validation error
 */
router.post(
  '/', // POST /api/discounts
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  discountValidations.validateCreateDiscount, // Validate request
  discountControllers.createDiscount // Create discount controller
);

/**
 * @swagger
 * /api/discounts:
 *   get:
 *     summary: List discounts
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 50
 *         description: Items per page
 *       - in: query
 *         name: active
 *         schema:
 *           type: boolean
 *         description: Filter by active status
 *       - in: query
 *         name: discount_type
 *         schema:
 *           type: string
 *           enum: [PERCENTAGE, FIXED_AMOUNT, BUY_X_GET_Y]
 *         description: Filter by discount type
 *       - in: query
 *         name: search
 *         schema:
 *           type: string
 *         description: Search by name or code
 *     responses:
 *       200:
 *         description: Discounts retrieved successfully
 */
router.get(
  '/', // GET /api/discounts
  authenticate, // Require authentication
  discountValidations.validateListDiscounts, // Validate request
  discountControllers.listDiscounts // List discounts controller
);

/**
 * @swagger
 * /api/discounts/validate:
 *   post:
 *     summary: Validate discount
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - discount_code or discount_id
 *             properties:
 *               discount_code:
 *                 type: string
 *                 description: Discount code
 *               discount_id:
 *                 type: integer
 *                 description: Discount ID
 *               customer_id:
 *                 type: integer
 *                 description: Customer ID (for per-customer limits)
 *     responses:
 *       200:
 *         description: Discount validation result
 *       400:
 *         description: Validation error
 */
router.post(
  '/validate', // POST /api/discounts/validate
  authenticate, // Require authentication
  discountValidations.validateValidateDiscount, // Validate request
  discountControllers.validateDiscount // Validate discount controller
);

/**
 * @swagger
 * /api/discounts/apply:
 *   post:
 *     summary: Apply discount (preview calculation)
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - discount_code or discount_id
 *               - sale_items
 *             properties:
 *               discount_code:
 *                 type: string
 *                 description: Discount code
 *               discount_id:
 *                 type: integer
 *                 description: Discount ID
 *               sale_items:
 *                 type: array
 *                 description: Sale items to apply discount to
 *                 items:
 *                   type: object
 *                   required:
 *                     - product_id
 *                     - quantity
 *                     - unit_price
 *                   properties:
 *                     product_id:
 *                       type: integer
 *                     quantity:
 *                       type: number
 *                     unit_price:
 *                       type: number
 *               customer_id:
 *                 type: integer
 *                 description: Customer ID
 *     responses:
 *       200:
 *         description: Discount applied successfully
 *       400:
 *         description: Validation error
 */
router.post(
  '/apply', // POST /api/discounts/apply
  authenticate, // Require authentication
  discountValidations.validateApplyDiscount, // Validate request
  discountControllers.applyDiscount // Apply discount controller
);

/**
 * @swagger
 * /api/discounts/{id}:
 *   get:
 *     summary: Get discount by ID
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *     responses:
 *       200:
 *         description: Discount retrieved successfully
 *       404:
 *         description: Discount not found
 */
router.get(
  '/:id', // GET /api/discounts/:id
  authenticate, // Require authentication
  discountValidations.validateGetDiscount, // Validate request
  discountControllers.getDiscount // Get discount controller
);

/**
 * @swagger
 * /api/discounts/code/{code}:
 *   get:
 *     summary: Get discount by code
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: code
 *         required: true
 *         schema:
 *           type: string
 *         description: Discount code
 *     responses:
 *       200:
 *         description: Discount retrieved successfully
 *       404:
 *         description: Discount not found
 */
router.get(
  '/code/:code', // GET /api/discounts/code/:code
  authenticate, // Require authentication
  discountValidations.validateGetDiscountByCode, // Validate request
  discountControllers.getDiscountByCode // Get discount by code controller
);

/**
 * @swagger
 * /api/discounts/product/{productId}:
 *   get:
 *     summary: Get discounts available for a product
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: productId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     responses:
 *       200:
 *         description: Available discounts retrieved successfully
 *       404:
 *         description: Product not found
 */
router.get(
  '/product/:productId', // GET /api/discounts/product/:productId
  authenticate, // Require authentication
  discountValidations.validateGetDiscountsForProduct, // Validate request
  discountControllers.getDiscountsForProduct // Get discounts for product controller
);

/**
 * @swagger
 * /api/discounts/{id}:
 *   put:
 *     summary: Update discount
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               code:
 *                 type: string
 *               discount_type:
 *                 type: string
 *                 enum: [PERCENTAGE, FIXED_AMOUNT, BUY_X_GET_Y]
 *               discount_value:
 *                 type: number
 *               min_purchase_amount:
 *                 type: number
 *               max_discount_amount:
 *                 type: number
 *               effective_from:
 *                 type: string
 *                 format: date-time
 *               effective_to:
 *                 type: string
 *                 format: date-time
 *               max_uses_per_customer:
 *                 type: integer
 *               max_total_uses:
 *                 type: integer
 *               active:
 *                 type: boolean
 *               description:
 *                 type: string
 *     responses:
 *       200:
 *         description: Discount updated successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Discount not found
 */
router.put(
  '/:id', // PUT /api/discounts/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  discountValidations.validateUpdateDiscount, // Validate request
  discountControllers.updateDiscount // Update discount controller
);

/**
 * @swagger
 * /api/discounts/{id}:
 *   delete:
 *     summary: Delete discount
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *     responses:
 *       200:
 *         description: Discount deleted successfully
 *       404:
 *         description: Discount not found
 */
router.delete(
  '/:id', // DELETE /api/discounts/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  discountValidations.validateDeleteDiscount, // Validate request
  discountControllers.deleteDiscount // Delete discount controller
);

// ============================================
// Discount Rule Routes
// ============================================

/**
 * @swagger
 * /api/discounts/{discountId}/rules:
 *   post:
 *     summary: Create discount rule
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: discountId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - rule_type
 *             properties:
 *               rule_type:
 *                 type: string
 *                 enum: [PRODUCT, CATEGORY, CUSTOMER, ALL]
 *                 description: Rule type
 *               product_id:
 *                 type: integer
 *                 description: Product ID (required if rule_type is PRODUCT)
 *               category_id:
 *                 type: integer
 *                 description: Category ID (required if rule_type is CATEGORY)
 *               customer_id:
 *                 type: integer
 *                 description: Customer ID (required if rule_type is CUSTOMER)
 *               min_quantity:
 *                 type: number
 *                 description: Minimum quantity
 *     responses:
 *       201:
 *         description: Discount rule created successfully
 *       400:
 *         description: Validation error
 */
router.post(
  '/:discountId/rules', // POST /api/discounts/:discountId/rules
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  discountValidations.validateCreateDiscountRule, // Validate request
  discountControllers.createDiscountRule // Create discount rule controller
);

/**
 * @swagger
 * /api/discounts/{discountId}/rules:
 *   get:
 *     summary: List discount rules
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: discountId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *     responses:
 *       200:
 *         description: Discount rules retrieved successfully
 */
router.get(
  '/:discountId/rules', // GET /api/discounts/:discountId/rules
  authenticate, // Require authentication
  discountValidations.validateListDiscountRules, // Validate request
  discountControllers.listDiscountRules // List discount rules controller
);

/**
 * @swagger
 * /api/discounts/{discountId}/rules/{ruleId}:
 *   get:
 *     summary: Get discount rule by ID
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: discountId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *       - in: path
 *         name: ruleId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Rule ID
 *     responses:
 *       200:
 *         description: Discount rule retrieved successfully
 *       404:
 *         description: Discount rule not found
 */
router.get(
  '/:discountId/rules/:ruleId', // GET /api/discounts/:discountId/rules/:ruleId
  authenticate, // Require authentication
  discountValidations.validateGetDiscountRule, // Validate request
  discountControllers.getDiscountRule // Get discount rule controller
);

/**
 * @swagger
 * /api/discounts/{discountId}/rules/{ruleId}:
 *   put:
 *     summary: Update discount rule
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: discountId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *       - in: path
 *         name: ruleId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Rule ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               rule_type:
 *                 type: string
 *                 enum: [PRODUCT, CATEGORY, CUSTOMER, ALL]
 *               product_id:
 *                 type: integer
 *               category_id:
 *                 type: integer
 *               customer_id:
 *                 type: integer
 *               min_quantity:
 *                 type: number
 *     responses:
 *       200:
 *         description: Discount rule updated successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Discount rule not found
 */
router.put(
  '/:discountId/rules/:ruleId', // PUT /api/discounts/:discountId/rules/:ruleId
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  discountValidations.validateUpdateDiscountRule, // Validate request
  discountControllers.updateDiscountRule // Update discount rule controller
);

/**
 * @swagger
 * /api/discounts/{discountId}/rules/{ruleId}:
 *   delete:
 *     summary: Delete discount rule
 *     tags: [Discounts]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: discountId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Discount ID
 *       - in: path
 *         name: ruleId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Rule ID
 *     responses:
 *       200:
 *         description: Discount rule deleted successfully
 *       404:
 *         description: Discount rule not found
 */
router.delete(
  '/:discountId/rules/:ruleId', // DELETE /api/discounts/:discountId/rules/:ruleId
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  discountValidations.validateDeleteDiscountRule, // Validate request
  discountControllers.deleteDiscountRule // Delete discount rule controller
);

// Export router
module.exports = router;

