/**
 * Price List Model
 * Represents different price lists (retail, wholesale, B2B, etc.)
 */

module.exports = (sequelize, DataTypes) => {
  // Define PriceList model
  const PriceList = sequelize.define('PriceList', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Price list name (e.g., "Retail", "Wholesale", "B2B")
    name: {
      type: DataTypes.STRING(100), // Name with max length
      allowNull: false, // Name is required
      validate: {
        notEmpty: true, // Name cannot be empty string
        len: [1, 100], // Length validation
      },
    },
    // Price list code (e.g., "RETAIL", "WHOLESALE", "B2B")
    code: {
      type: DataTypes.STRING(50), // Code with max length
      allowNull: false, // Code is required
      unique: true, // Price list code must be unique
      validate: {
        notEmpty: true, // Code cannot be empty string
        len: [1, 50], // Length validation
      },
    },
    // Price list type (RETAIL, WHOLESALE, B2B, CUSTOMER_SPECIFIC)
    price_list_type: {
      type: DataTypes.ENUM('RETAIL', 'WHOLESALE', 'B2B', 'CUSTOMER_SPECIFIC'), // Enum for price list type
      allowNull: false, // Price list type is required
      validate: {
        isIn: [['RETAIL', 'WHOLESALE', 'B2B', 'CUSTOMER_SPECIFIC']], // Validate enum values
      },
    },
    // Currency code (e.g., "KES", "USD")
    currency: {
      type: DataTypes.STRING(3), // Currency code (ISO 4217)
      allowNull: false, // Currency is required
      defaultValue: 'KES', // Default to Kenya Shilling
      validate: {
        len: [3, 3], // Currency code must be exactly 3 characters
      },
    },
    // Whether this is the default price list
    is_default: {
      type: DataTypes.BOOLEAN, // Boolean for default flag
      defaultValue: false, // Default to not default
      allowNull: false, // Default flag is required
    },
    // Whether this price list is active
    active: {
      type: DataTypes.BOOLEAN, // Boolean for active status
      defaultValue: true, // Default to active
      allowNull: false, // Active status is required
    },
    // Description of the price list
    description: {
      type: DataTypes.TEXT, // Text field for description
      allowNull: true, // Description is optional
    },
  }, {
    // Model options
    tableName: 'price_lists', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: true, // Enable createdAt and updatedAt timestamps
    createdAt: 'created_at', // Map createdAt to created_at column
    updatedAt: 'updated_at', // Map updatedAt to updated_at column
    indexes: [
      // Unique index on code (already enforced by unique constraint)
      {
        unique: true, // Unique index
        fields: ['code'], // Index on code field
      },
      // Index on price_list_type for filtering
      {
        fields: ['price_list_type'], // Index on price_list_type field
      },
      // Index on is_default for finding default price list
      {
        fields: ['is_default'], // Index on is_default field
      },
      // Index on active for filtering
      {
        fields: ['active'], // Index on active field
      },
    ],
  });

  // Define model associations
  PriceList.associate = (models) => {
    // PriceList has many ProductPrices (one-to-many relationship)
    PriceList.hasMany(models.ProductPrice, {
      foreignKey: 'price_list_id', // Foreign key in ProductPrices table
      as: 'productPrices', // Alias for association
    });
    
    // PriceList has many Customers (one-to-many relationship) - for customer-specific pricing
    PriceList.hasMany(models.Customer, {
      foreignKey: 'price_list_id', // Foreign key in Customers table
      as: 'customers', // Alias for association
    });
  };

  // Return PriceList model
  return PriceList;
};

