/**
 * GRN Item Model
 * Represents items received in a GRN
 */

module.exports = (sequelize, DataTypes) => {
  // Define GRNItem model
  const GRNItem = sequelize.define('GRNItem', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Foreign key to GRN
    grn_id: {
      type: DataTypes.BIGINT, // Match GRN ID type
      allowNull: false, // GRN ID is required
      references: {
        model: 'grns', // Reference to grns table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on GRN update
      onDelete: 'CASCADE', // Cascade delete when GRN is deleted
    },
    // Foreign key to Product
    product_id: {
      type: DataTypes.BIGINT, // Match Product ID type
      allowNull: false, // Product ID is required
      references: {
        model: 'products', // Reference to products table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on product update
    },
    // Quantity received
    quantity: {
      type: DataTypes.DECIMAL(12, 3), // Decimal with 12 digits, 3 decimal places
      allowNull: false, // Quantity is required
      validate: {
        min: 0, // Quantity cannot be negative
      },
    },
    // Unit cost per item
    unit_cost: {
      type: DataTypes.DECIMAL(12, 2), // Decimal with 12 digits, 2 decimal places
      allowNull: false, // Unit cost is required
      validate: {
        min: 0, // Unit cost cannot be negative
      },
    },
    // Dimension fields for RM products (added in migration 010)
    piece_length: {
      type: DataTypes.DECIMAL(12, 3), // Length of each piece received
      allowNull: true, // Optional - only for RM products with dimensions
      validate: {
        min: 0, // Length cannot be negative
      },
    },
    piece_width: {
      type: DataTypes.DECIMAL(12, 3), // Width of each piece received
      allowNull: true, // Optional - only for RM products with dimensions
      validate: {
        min: 0, // Width cannot be negative
      },
    },
    dimension_unit: {
      type: DataTypes.ENUM('inch', 'cm', 'm'), // Unit of measure for piece dimensions
      allowNull: true, // Optional - only for RM products with dimensions
    },
    pieces_count: {
      type: DataTypes.INTEGER, // Number of pieces with these dimensions
      allowNull: false, // Required
      defaultValue: 1, // Default to 1 piece
      validate: {
        min: 1, // Must have at least 1 piece
      },
    },
  }, {
    // Model options
    tableName: 'grn_items', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: false, // No timestamps for GRN items
    indexes: [
      // Index on grn_id for faster lookups
      {
        fields: ['grn_id'], // Index on grn_id field
      },
      // Index on product_id for faster lookups
      {
        fields: ['product_id'], // Index on product_id field
      },
    ],
  });

  // Define model associations
  GRNItem.associate = (models) => {
    // GRNItem belongs to GRN (many-to-one relationship)
    GRNItem.belongsTo(models.GRN, {
      foreignKey: 'grn_id', // Foreign key in GRNItem table
      as: 'grn', // Alias for association
    });
    
    // GRNItem belongs to Product (many-to-one relationship)
    GRNItem.belongsTo(models.Product, {
      foreignKey: 'product_id', // Foreign key in GRNItem table
      as: 'product', // Alias for association
    });
    
  };

  // Return GRNItem model
  return GRNItem;
};

