# XYZ POS - Deployment Guide (Namecheap + Docker + GitHub Actions)

This guide covers deploying the **client** and **server** to Namecheap VPS using Docker and GitHub Actions CI/CD. Triggered only on **push to main**.

---

## Prerequisites

1. **Namecheap VPS** (shared hosting does not support Docker)
2. **GitHub repository** with this codebase
3. **MySQL database** (Namecheap MySQL, or MySQL on VPS, or external provider)
4. **Domain** pointing to your VPS IP (optional but recommended)

---

## Architecture

```
Push to main → GitHub Actions
  → Build Docker images (client, server)
  → Push to ghcr.io (GitHub Container Registry)
  → SSH to Namecheap VPS
  → Pull images & run docker compose
```

- **Client**: Next.js on port 3000
- **Server**: Express API on port 4000
- **PrintService**: NOT deployed. Users install locally (Windows) for label printing.

---

## 1. GitHub Secrets

Add these in: **Settings → Secrets and variables → Actions → New repository secret**

### Required for deployment

| Secret | Description | Example |
|--------|-------------|---------|
| `NAMECHEAP_SSH_HOST` | VPS hostname or IP | `vps123.namecheap.com` or `192.168.1.100` |
| `NAMECHEAP_SSH_USER` | SSH username | `root` or `deploy` |
| `NAMECHEAP_SSH_KEY` | SSH private key (full content) | `-----BEGIN OPENSSH PRIVATE KEY-----...` |
| `NAMECHEAP_SSH_PORT` | SSH port (optional, default 22) | `22` |
| `NAMECHEAP_DEPLOY_PATH` | Path on VPS where repo is cloned | `/opt/xyz-pos` |
| `GHCR_TOKEN` | GitHub PAT with `read:packages` scope | `ghp_xxxx` |
| `NAMECHEAP_GHCR_USER` | GitHub username for registry login | Your GitHub username |

### Required for frontend build (baked at build time)

| Secret | Description | Example |
|--------|-------------|---------|
| `NEXT_PUBLIC_API_URL` | Full API URL (browser-facing) | `https://api.yourdomain.com/api` |
| `NEXT_PUBLIC_PRINT_SERVICE_URL` | (Optional) Local print service; leave default for POS | `http://127.0.0.1:9101` |
| `NEXT_PUBLIC_PRINT_SERVICE_KEY` | (Optional) Print service API key | `xyz-pos-local-token` |

### Server environment (create on VPS, not GitHub Secrets)

Create `server/.env.production` **on the VPS** at `$NAMECHEAP_DEPLOY_PATH/server/.env.production`:

```env
NODE_ENV=production
HOST=0.0.0.0
PORT=4000

JWT_SECRET="your-strong-secret-min-32-chars"
JWT_EXPIRES_IN=24h
JWT_REFRESH_EXPIRES_IN=7d

DB_DIALECT=mysql
DB_HOST="your-db-host"
DB_PORT=3306
DB_USER="your-db-user"
DB_PASSWORD="your-db-password"
DB_NAME="xyz_pos_db"
DB_LOGGING=false

PAYSTACK_SECRET_KEY=""
PAYSTACK_PUBLIC_KEY=""
PAYSTACK_BASE_URL=https://api.paystack.co

BARCODE_PREFIX=XYZ
BARCODE_LENGTH=12

MAX_FILE_SIZE=5242880
ALLOWED_FILE_TYPES=image/jpeg,image/png,image/jpg
UPLOAD_PATH=./uploads

RATE_LIMIT_WINDOW_MS=900000
RATE_LIMIT_MAX=100

CORS_ORIGIN="https://your-domain.com"
```

---

## 2. VPS Setup (One-time)

### 2.1 Install Docker on Namecheap VPS

```bash
# SSH into your VPS
ssh user@your-vps-ip

# Install Docker (Ubuntu/Debian)
curl -fsSL https://get.docker.com | sh
sudo usermod -aG docker $USER
# Log out and back in for group to apply
```

### 2.2 Clone repo and create env

```bash
sudo mkdir -p /opt/xyz-pos
sudo chown $USER:$USER /opt/xyz-pos
cd /opt/xyz-pos
git clone https://github.com/YOUR_USERNAME/YOUR_REPO.git .

mkdir -p server
# Create server/.env.production (paste content from above)
nano server/.env.production
```

### 2.3 Ensure packages are public (or configure PAT)

For **public** packages: In GitHub → Settings → Packages → Package visibility, set to public.  
For **private** packages: Use a PAT with `read:packages` as `GHCR_TOKEN`.

---

## 3. Database Setup

Use Namecheap MySQL, or install MySQL on the VPS:

```bash
# Example: MySQL on same VPS
sudo apt install mysql-server
sudo mysql_secure_installation
mysql -u root -p -e "CREATE DATABASE xyz_pos_db; CREATE USER 'xyzuser'@'localhost' IDENTIFIED BY 'strongpassword'; GRANT ALL ON xyz_pos_db.* TO 'xyzuser'@'localhost'; FLUSH PRIVILEGES;"
```

Update `DB_HOST`, `DB_USER`, `DB_PASSWORD` in `server/.env.production`.

---

## 4. First Deploy

Push to `main`:

```bash
git push origin main
```

This triggers the workflow. Check: **Actions** tab → latest run.

---

## 5. Reverse Proxy (Nginx, optional)

To serve on ports 80/443 with SSL:

```nginx
# /etc/nginx/sites-available/xyz-pos
server {
    listen 80;
    server_name your-domain.com api.your-domain.com;

    location / {
        proxy_pass http://127.0.0.1:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
    }

    location /api {
        proxy_pass http://127.0.0.1:4000;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
    }
}
```

Enable: `sudo ln -s /etc/nginx/sites-available/xyz-pos /etc/nginx/sites-enabled/`  
Reload: `sudo nginx -t && sudo systemctl reload nginx`

---

## 6. PrintService (Local install, not deployed)

PrintService runs only on the **client machine** (POS terminals) for label printing. Users must:

1. Download the .NET PrintService build (or build from source)
2. Install as Windows Service (see `PrintService/README.md`)
3. Ensure PrintService URL in the app points to `http://127.0.0.1:9101` (default)

The packaged installer / standalone executable will be available separately. Build from `PrintService/` with:

```bat
cd PrintService
dotnet publish -c Release -r win-x64 --self-contained
```

---

## 7. Troubleshooting

| Issue | Solution |
|-------|----------|
| `docker login` fails | Verify `GHCR_TOKEN` and `NAMECHEAP_GHCR_USER`; ensure PAT has `read:packages` |
| `Connection refused` on deploy | Check `NAMECHEAP_SSH_HOST`, `NAMECHEAP_SSH_PORT`, firewall |
| Server container exits | Check `server/.env.production`, DB connectivity, logs: `docker logs xyz-pos-server` |
| CORS errors | Set `CORS_ORIGIN` to your exact frontend URL (with https) |
| Images not found | Ensure `GHCR_IMAGE_PREFIX` matches `ghcr.io/YOUR_GITHUB_USERNAME` |

---

## Summary

- **Client + Server**: Deployed via Docker on Namecheap VPS
- **CI/CD**: GitHub Actions on push to `main`
- **Secrets**: Stored in GitHub Secrets; server env on VPS
- **PrintService**: Installed locally on each POS device; not part of cloud deployment
