/**
 * Taxes API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const taxesAPI = {
  /**
   * List tax rates
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  listTaxRates: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.TAXES.TAX_RATES, { params });
  },

  /**
   * Get tax rate by ID
   * @param {number} id
   * @returns {Promise}
   */
  getTaxRate: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.TAXES.TAX_RATES}/${id}`);
  },

  /**
   * Get tax rate by code
   * @param {string} code
   * @returns {Promise}
   */
  getTaxRateByCode: async (code) => {
    return apiClient.get(`${API_ENDPOINTS.TAXES.TAX_RATES}/code/${code}`);
  },

  /**
   * List tax categories
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  listTaxCategories: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.TAXES.TAX_CATEGORIES, { params });
  },

  /**
   * Get tax category by ID
   * @param {number} id
   * @returns {Promise}
   */
  getTaxCategory: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.TAXES.TAX_CATEGORIES}/${id}`);
  },

  /**
   * Create tax rate
   * @param {Object} taxRateData
   * @returns {Promise}
   */
  createTaxRate: async (taxRateData) => {
    return apiClient.post(API_ENDPOINTS.TAXES.TAX_RATES, taxRateData);
  },

  /**
   * Update tax rate
   * @param {number} id
   * @param {Object} taxRateData
   * @returns {Promise}
   */
  updateTaxRate: async (id, taxRateData) => {
    return apiClient.put(`${API_ENDPOINTS.TAXES.TAX_RATES}/${id}`, taxRateData);
  },

  /**
   * Delete tax rate
   * @param {number} id
   * @returns {Promise}
   */
  deleteTaxRate: async (id) => {
    return apiClient.delete(`${API_ENDPOINTS.TAXES.TAX_RATES}/${id}`);
  },

  /**
   * Create tax category
   * @param {Object} taxCategoryData
   * @returns {Promise}
   */
  createTaxCategory: async (taxCategoryData) => {
    return apiClient.post(API_ENDPOINTS.TAXES.TAX_CATEGORIES, taxCategoryData);
  },

  /**
   * Update tax category
   * @param {number} id
   * @param {Object} taxCategoryData
   * @returns {Promise}
   */
  updateTaxCategory: async (id, taxCategoryData) => {
    return apiClient.put(`${API_ENDPOINTS.TAXES.TAX_CATEGORIES}/${id}`, taxCategoryData);
  },

  /**
   * Delete tax category
   * @param {number} id
   * @returns {Promise}
   */
  deleteTaxCategory: async (id) => {
    return apiClient.delete(`${API_ENDPOINTS.TAXES.TAX_CATEGORIES}/${id}`);
  },

  /**
   * List tax exemptions
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  listTaxExemptions: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.TAXES.TAX_EXEMPTIONS, { params });
  },

  /**
   * Get tax exemption by ID
   * @param {number} id
   * @returns {Promise}
   */
  getTaxExemption: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.TAXES.TAX_EXEMPTIONS}/${id}`);
  },

  /**
   * Create tax exemption
   * @param {Object} taxExemptionData
   * @returns {Promise}
   */
  createTaxExemption: async (taxExemptionData) => {
    return apiClient.post(API_ENDPOINTS.TAXES.TAX_EXEMPTIONS, taxExemptionData);
  },

  /**
   * Update tax exemption
   * @param {number} id
   * @param {Object} taxExemptionData
   * @returns {Promise}
   */
  updateTaxExemption: async (id, taxExemptionData) => {
    return apiClient.put(`${API_ENDPOINTS.TAXES.TAX_EXEMPTIONS}/${id}`, taxExemptionData);
  },

  /**
   * Delete tax exemption
   * @param {number} id
   * @returns {Promise}
   */
  deleteTaxExemption: async (id) => {
    return apiClient.delete(`${API_ENDPOINTS.TAXES.TAX_EXEMPTIONS}/${id}`);
  },
};

