/**
 * Products API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const productsAPI = {
  /**
   * List products
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  list: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.PRODUCTS, { params });
  },

  /**
   * Get product by ID
   * @param {number} id
   * @returns {Promise}
   */
  getById: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTS}/${id}`);
  },

  /**
   * Get product stock availability
   * @param {number} id - Product ID
   * @param {number} variantId - Optional variant ID
   * @returns {Promise}
   */
  getAvailability: async (id, variantId = null) => {
    const params = variantId ? { variant_id: variantId } : {};
    return apiClient.get(`${API_ENDPOINTS.PRODUCTS}/${id}/availability`, { params });
  },

  /**
   * Search products
   * @param {string} query
   * @param {Object} params
   * @returns {Promise}
   */
  search: async (query, params = {}) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTS}/search`, {
      params: { q: query, ...params },
    });
  },

  /**
   * Create product
   * @param {Object} productData
   * @returns {Promise}
   */
  create: async (productData) => {
    return apiClient.post(API_ENDPOINTS.PRODUCTS, productData);
  },

  /**
   * Update product
   * @param {number} id
   * @param {Object} productData
   * @returns {Promise}
   */
  update: async (id, productData) => {
    return apiClient.put(`${API_ENDPOINTS.PRODUCTS}/${id}`, productData);
  },

  /**
   * Get product variants
   * @param {number} productId
   * @returns {Promise}
   */
  getVariants: async (productId) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTS}/${productId}/variants`);
  },

  /**
   * Create product variant
   * @param {number} productId
   * @param {Object} variantData
   * @returns {Promise}
   */
  createVariant: async (productId, variantData) => {
    return apiClient.post(`${API_ENDPOINTS.PRODUCTS}/${productId}/variants`, variantData);
  },

  /**
   * Update product variant
   * @param {number} productId
   * @param {number} variantId
   * @param {Object} variantData
   * @returns {Promise}
   */
  updateVariant: async (productId, variantId, variantData) => {
    return apiClient.put(`${API_ENDPOINTS.PRODUCTS}/${productId}/variants/${variantId}`, variantData);
  },

  /**
   * Delete product variant
   * @param {number} productId
   * @param {number} variantId
   * @returns {Promise}
   */
  deleteVariant: async (productId, variantId) => {
    return apiClient.delete(`${API_ENDPOINTS.PRODUCTS}/${productId}/variants/${variantId}`);
  },
};

