/**
 * Label Printing API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const labelPrintingAPI = {
  /**
   * Get ZPL data for inventory item (client-side printing)
   * @param {number} inventoryItemId - Inventory item ID
   * @param {Object} options - Optional parameters (dpi, labelLength, labelWidth)
   * @returns {Promise} ZPL data
   */
  getInventoryItemLabelZPL: async (inventoryItemId, options = {}) => {
    const params = {};
    if (options.dpi) params.dpi = options.dpi;
    if (options.labelLength) params.labelLength = options.labelLength;
    if (options.labelWidth) params.labelWidth = options.labelWidth;
    return apiClient.get(`${API_ENDPOINTS.LABEL_PRINTING}/inventory-items/${inventoryItemId}/zpl`, { params });
  },

  /**
   * Print label for inventory item (server-side - for network printers only)
   * @param {number} inventoryItemId - Inventory item ID
   * @param {Object} printerConfig - Printer configuration
   * @param {string} printerConfig.method - Printer method ('network' or 'usb')
   * @param {string} printerConfig.host - Printer host (for network printing)
   * @param {number} printerConfig.port - Printer port (for network printing, default: 9100)
   * @param {string} printerConfig.devicePath - Printer device path (for USB printing, e.g., /dev/usb/lp0)
   * @returns {Promise}
   */
  printInventoryItemLabel: async (inventoryItemId, printerConfig = {}) => {
    return apiClient.post(`${API_ENDPOINTS.LABEL_PRINTING}/inventory-items/${inventoryItemId}/print`, {
      printer: printerConfig,
    });
  },

  /**
   * Get ZPL data for product (client-side printing)
   * @param {number} productId - Product ID
   * @param {string} sampleUID - Sample UID to use (optional)
   * @param {Object} options - Optional parameters (dpi, labelLength, labelWidth)
   * @returns {Promise} ZPL data
   */
  getProductLabelZPL: async (productId, sampleUID = null, options = {}) => {
    const params = {};
    if (sampleUID) params.uid = sampleUID;
    if (options.dpi) params.dpi = options.dpi;
    if (options.labelLength) params.labelLength = options.labelLength;
    if (options.labelWidth) params.labelWidth = options.labelWidth;
    return apiClient.get(`${API_ENDPOINTS.LABEL_PRINTING}/products/${productId}/zpl`, { params });
  },

  /**
   * Print label for product (server-side - for network printers only)
   * @param {number} productId - Product ID
   * @param {Object} options - Options (optional)
   * @param {Object} options.printer - Printer configuration (optional, for actual printing)
   * @param {string} options.sampleUID - Sample UID to use (optional)
   * @returns {Promise}
   */
  printProductLabel: async (productId, options = {}) => {
    return apiClient.post(`${API_ENDPOINTS.LABEL_PRINTING}/products/${productId}/print`, {
      printer: options.printer,
      sampleUID: options.sampleUID,
    });
  },

  /**
   * Generate label preview (QR code image)
   * @param {Object} params - Parameters (optional)
   * @param {number} params.inventory_item_id - Inventory item ID (optional, if product_id not provided)
   * @param {number} params.product_id - Product ID (optional, if inventory_item_id not provided)
   * @param {string} params.uid - UID to use (optional)
   * @returns {Promise}
   */
  generateLabelPreview: async (params = {}) => {
    return apiClient.get(`${API_ENDPOINTS.LABEL_PRINTING}/preview`, { params });
  },

  /**
   * Generate ZPL template (without printing)
   * @param {Object} data - Label data
   * @param {string} data.productName - Product name
   * @param {string} data.uid - UID for the label
   * @param {string} data.qrData - Data to encode in QR code (optional, defaults to uid)
   * @returns {Promise}
   */
  generateZPLTemplate: async (data) => {
    return apiClient.post(`${API_ENDPOINTS.LABEL_PRINTING}/zpl`, data);
  },

  /**
   * Generate ZPL for a batch of inventory items (client-side/local printing)
   * @param {Array<number>} inventoryItemIds - List of inventory item IDs
   * @param {Object} options - Optional parameters (dpi, labelLength, labelWidth)
   * @returns {Promise<{ items: Array<{ inventoryItemId, productName, uid, zplData }> }>}
   */
  generateBatchInventoryItemZPL: async (inventoryItemIds, options = {}) => {
    const payload = {
      inventory_item_ids: inventoryItemIds,
    };

    if (options.dpi) payload.dpi = options.dpi;
    if (options.labelLength) payload.labelLength = options.labelLength;
    if (options.labelWidth) payload.labelWidth = options.labelWidth;

    return apiClient.post(
      `${API_ENDPOINTS.LABEL_PRINTING}/inventory-items/batch-zpl`,
      payload,
    );
  },
};
