/**
 * Inventory API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const inventoryAPI = {
  /**
   * List inventory
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  list: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.INVENTORY, { params });
  },

  /**
   * Get inventory by product ID
   * @param {number} productId
   * @param {number} variantId - Optional variant ID
   * @returns {Promise}
   */
  getByProduct: async (productId, variantId = null) => {
    const params = variantId ? { variant_id: variantId } : {};
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/${productId}`, { params });
  },

  /**
   * Get inventory quantity
   * @param {number} productId
   * @param {number} variantId - Optional variant ID
   * @returns {Promise}
   */
  getQuantity: async (productId, variantId = null) => {
    const params = { product_id: productId };
    if (variantId) params.variant_id = variantId;
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/quantity`, { params });
  },

  /**
   * Check availability
   * @param {number} productId
   * @param {number} quantity
   * @param {number} variantId - Optional variant ID
   * @returns {Promise}
   */
  checkAvailability: async (productId, quantity, variantId = null) => {
    const params = { product_id: productId, quantity };
    if (variantId) params.variant_id = variantId;
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/check-availability`, { params });
  },

  /**
   * Get inventory movements
   * @param {number} productId
   * @param {number} variantId - Optional variant ID
   * @param {Object} params - Pagination params
   * @returns {Promise}
   */
  getMovements: async (productId, variantId = null, params = {}) => {
    const queryParams = { ...params };
    if (variantId) queryParams.variant_id = variantId;
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/${productId}/movements`, { params: queryParams });
  },

  /**
   * Adjust inventory
   * @param {Object} adjustmentData
   * @returns {Promise}
   */
  adjust: async (adjustmentData) => {
    return apiClient.post(`${API_ENDPOINTS.INVENTORY}/adjust`, adjustmentData);
  },

  /**
   * Set inventory quantity
   * @param {Object} quantityData
   * @returns {Promise}
   */
  setQuantity: async (quantityData) => {
    return apiClient.post(`${API_ENDPOINTS.INVENTORY}/set-quantity`, quantityData);
  },

  /**
   * Set reorder level
   * @param {Object} reorderData
   * @returns {Promise}
   */
  setReorderLevel: async (reorderData) => {
    return apiClient.post(`${API_ENDPOINTS.INVENTORY}/set-reorder-level`, reorderData);
  },

  /**
   * Get reconciliation report
   * @param {number} productId - Optional product ID
   * @returns {Promise}
   */
  getReconciliationReport: async (productId = null) => {
    const params = productId ? { product_id: productId } : {};
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/reconciliation/report`, { params });
  },

  /**
   * Check for discrepancies
   * @param {number} productId - Optional product ID
   * @returns {Promise}
   */
  checkDiscrepancies: async (productId = null) => {
    const params = productId ? { product_id: productId } : {};
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/reconciliation/check`, { params });
  },

  /**
   * Auto-reconcile discrepancies
   * @param {number} productId - Optional product ID
   * @returns {Promise}
   */
  autoReconcile: async (productId = null) => {
    const body = productId ? { product_id: productId } : {};
    return apiClient.post(`${API_ENDPOINTS.INVENTORY}/reconciliation/auto-reconcile`, body);
  },

  /**
   * Get inventory item by UID or barcode
   * @param {string} uid - UID or barcode
   * @returns {Promise}
   */
  getItemByUID: async (uid) => {
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/items/${uid}`);
  },

  /**
   * Scan inventory item by UID or barcode
   * @param {string} uid - UID or barcode
   * @returns {Promise}
   */
  scanItem: async (uid) => {
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/items/${uid}/scan`);
  },

  /**
   * List inventory items (UID-based)
   * @param {Object} params - Query parameters (product_id, variant_id, status, page, limit)
   * @returns {Promise}
   */
  listItems: async (params = {}) => {
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/items`, { params });
  },

  /**
   * List RM inventory pieces (dimension-based)
   * @param {number} productId - Product ID
   * @param {Object} params - Query parameters (status, page, limit, min_length, max_length, min_width, max_width, unit)
   * @returns {Promise}
   */
  listRMPieces: async (productId, params = {}) => {
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/rm/${productId}/pieces`, { params });
  },

  /**
   * Get RM inventory summary (dimension-based)
   * @param {number} productId - Product ID
   * @param {Object} params - Query parameters (unit)
   * @returns {Promise}
   */
  getRMSummary: async (productId, params = {}) => {
    return apiClient.get(`${API_ENDPOINTS.INVENTORY}/rm/${productId}/summary`, { params });
  },
};

