/**
 * Categories API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const categoriesAPI = {
  /**
   * List categories
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  list: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.CATEGORIES, { params });
  },

  /**
   * Get category by ID
   * @param {number} id
   * @returns {Promise}
   */
  getById: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.CATEGORIES}/${id}`);
  },

  /**
   * Get category hierarchy
   * @param {Object} params
   * @returns {Promise}
   */
  getHierarchy: async (params = {}) => {
    return apiClient.get(`${API_ENDPOINTS.CATEGORIES}/hierarchy`, { params });
  },

  /**
   * Create category
   * @param {Object} categoryData
   * @returns {Promise}
   */
  create: async (categoryData) => {
    return apiClient.post(API_ENDPOINTS.CATEGORIES, categoryData);
  },

  /**
   * Update category
   * @param {number} id
   * @param {Object} categoryData
   * @returns {Promise}
   */
  update: async (id, categoryData) => {
    return apiClient.put(`${API_ENDPOINTS.CATEGORIES}/${id}`, categoryData);
  },

  /**
   * Get product categories
   * @param {number} productId
   * @returns {Promise}
   */
  getProductCategories: async (productId) => {
    return apiClient.get(`${API_ENDPOINTS.CATEGORIES}/products/${productId}`);
  },

  /**
   * Assign product to categories
   * @param {number} productId
   * @param {Array} categoryIds
   * @param {number} primaryCategoryId
   * @returns {Promise}
   */
  assignProductToCategories: async (productId, categoryIds, primaryCategoryId = null) => {
    return apiClient.post(`${API_ENDPOINTS.CATEGORIES}/products/${productId}`, {
      category_ids: categoryIds,
      primary_category_id: primaryCategoryId,
    });
  },

  /**
   * Remove product from category
   * @param {number} productId
   * @param {number} categoryId
   * @returns {Promise}
   */
  removeProductFromCategory: async (productId, categoryId) => {
    return apiClient.delete(`${API_ENDPOINTS.CATEGORIES}/products/${productId}/${categoryId}`);
  },
};

