/**
 * Button Component
 * Pure Tailwind CSS - Modern POS buttons
 */

export default function Button({
  children,
  variant = 'primary',
  size = 'md',
  type = 'button',
  disabled = false,
  loading = false,
  onClick,
  className = '',
  ...props
}) {
  // Base styles - pure Tailwind
  const baseStyles = [
    'inline-flex items-center justify-center',
    'font-medium rounded-lg',
    'transition-all duration-200',
    'focus:outline-none focus:ring-2 focus:ring-offset-2',
    'disabled:opacity-50 disabled:cursor-not-allowed disabled:pointer-events-none',
    'active:scale-95',
  ].join(' ');

  // Variants - pure Tailwind colors
  const variants = {
    primary: [
      'bg-blue-600 text-white',
      'hover:bg-blue-700',
      'focus:ring-blue-500',
      'shadow-sm hover:shadow-md',
    ].join(' '),
    secondary: [
      'bg-gray-200 text-gray-900',
      'hover:bg-gray-300',
      'dark:bg-gray-700 dark:text-gray-100 dark:hover:bg-gray-600',
      'focus:ring-gray-500',
      'shadow-sm hover:shadow',
    ].join(' '),
    success: [
      'bg-green-600 text-white',
      'hover:bg-green-700',
      'focus:ring-green-500',
      'shadow-sm hover:shadow-md',
    ].join(' '),
    danger: [
      'bg-red-600 text-white',
      'hover:bg-red-700',
      'focus:ring-red-500',
      'shadow-sm hover:shadow-md',
    ].join(' '),
    warning: [
      'bg-yellow-500 text-white',
      'hover:bg-yellow-600',
      'focus:ring-yellow-500',
      'shadow-sm hover:shadow-md',
    ].join(' '),
    outline: [
      'border-2 border-gray-300 text-gray-700',
      'bg-transparent',
      'hover:bg-gray-50',
      'dark:border-gray-600 dark:text-gray-300 dark:hover:bg-gray-800',
      'focus:ring-gray-500',
    ].join(' '),
    ghost: [
      'text-gray-700 bg-transparent',
      'hover:bg-gray-100',
      'dark:text-gray-300 dark:hover:bg-gray-800',
      'focus:ring-gray-500',
    ].join(' '),
  };

  // Sizes - pure Tailwind spacing
  const sizes = {
    sm: 'px-3 py-1.5 text-sm gap-1.5',
    md: 'px-4 py-2 text-base gap-2',
    lg: 'px-6 py-3 text-lg gap-2.5',
    xl: 'px-8 py-4 text-xl gap-3',
  };

  const variantClass = variants[variant] || variants.primary;
  const sizeClass = sizes[size] || sizes.md;

  return (
    <button
      type={type}
      disabled={disabled || loading}
      onClick={onClick}
      className={`${baseStyles} ${variantClass} ${sizeClass} ${className}`}
      {...props}
    >
      {loading && (
        <svg
          className="animate-spin h-4 w-4"
          xmlns="http://www.w3.org/2000/svg"
          fill="none"
          viewBox="0 0 24 24"
        >
          <circle
            className="opacity-25"
            cx="12"
            cy="12"
            r="10"
            stroke="currentColor"
            strokeWidth="4"
          ></circle>
          <path
            className="opacity-75"
            fill="currentColor"
            d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
          ></path>
        </svg>
      )}
      <span className={loading ? 'opacity-0' : 'opacity-100'}>
        {children}
      </span>
    </button>
  );
}
