'use client';

/**
 * Category Sidebar Component
 * Category filter for POS product grid - Modern cashier-friendly design
 */

import { useQuery } from '@tanstack/react-query';
import { categoriesAPI } from '@/lib/api';
import { LayoutGrid, Tag, ChevronRight } from 'lucide-react';

export default function CategorySidebar({ selectedCategoryId, onCategorySelect }) {
  const { data: categoriesData, isLoading } = useQuery({
    queryKey: ['categories'],
    queryFn: async () => {
      const response = await categoriesAPI.list({ limit: 100 });
      return response;
    },
  });

  // Handle different response formats
  const categories = 
    categoriesData?.data?.categories || 
    categoriesData?.categories || 
    categoriesData?.data || 
    [];

  return (
    <div className="h-full bg-white dark:bg-gray-900 rounded-xl shadow-sm border border-gray-200 dark:border-gray-800 flex flex-col overflow-hidden">
      {/* Header */}
      <div className="px-4 py-3 border-b border-gray-200 dark:border-gray-800 bg-gradient-to-r from-blue-600 to-indigo-600">
        <h3 className="font-bold text-white text-sm flex items-center gap-2">
          <Tag className="w-4 h-4" />
          Categories
        </h3>
      </div>

      {/* Categories List */}
      <div className="flex-1 overflow-y-auto p-2.5">
        {isLoading ? (
          <div className="space-y-2">
            {[1, 2, 3, 4].map((i) => (
              <div key={i} className="h-12 bg-gray-100 dark:bg-gray-800 rounded-xl animate-pulse" />
            ))}
          </div>
        ) : (
          <ul className="space-y-1">
            {/* All Products Option */}
            <li>
              <button
                onClick={() => onCategorySelect(null)}
                className={`
                  w-full flex items-center gap-3 px-4 py-3 rounded-xl text-sm font-semibold
                  transition-all duration-200 ease-out
                  ${
                    selectedCategoryId === null
                      ? 'bg-gradient-to-r from-blue-600 to-indigo-600 text-white shadow-lg shadow-blue-500/30 scale-[1.02]'
                      : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-800 hover:scale-[1.01]'
                  }
                `}
              >
                <div className={`
                  w-8 h-8 rounded-lg flex items-center justify-center
                  ${selectedCategoryId === null 
                    ? 'bg-white/20' 
                    : 'bg-blue-100 dark:bg-blue-900/30'
                  }
                `}>
                  <LayoutGrid className={`w-4 h-4 ${selectedCategoryId === null ? 'text-white' : 'text-blue-600 dark:text-blue-400'}`} />
                </div>
                <span className="flex-1 text-left">All Products</span>
                {selectedCategoryId === null && (
                  <ChevronRight className="w-4 h-4 text-white/70" />
                )}
              </button>
            </li>

            {/* Category Items */}
            {categories.map((category, index) => {
              const isSelected = selectedCategoryId === category.id;
              // Generate consistent colors based on index
              const colors = [
                { bg: 'bg-emerald-100 dark:bg-emerald-900/30', text: 'text-emerald-600 dark:text-emerald-400' },
                { bg: 'bg-purple-100 dark:bg-purple-900/30', text: 'text-purple-600 dark:text-purple-400' },
                { bg: 'bg-amber-100 dark:bg-amber-900/30', text: 'text-amber-600 dark:text-amber-400' },
                { bg: 'bg-rose-100 dark:bg-rose-900/30', text: 'text-rose-600 dark:text-rose-400' },
                { bg: 'bg-cyan-100 dark:bg-cyan-900/30', text: 'text-cyan-600 dark:text-cyan-400' },
                { bg: 'bg-orange-100 dark:bg-orange-900/30', text: 'text-orange-600 dark:text-orange-400' },
              ];
              const colorSet = colors[index % colors.length];

              return (
                <li key={category.id}>
                  <button
                    onClick={() => onCategorySelect(category.id)}
                    className={`
                      w-full flex items-center gap-3 px-4 py-3 rounded-xl text-sm font-semibold
                      transition-all duration-200 ease-out
                      ${
                        isSelected
                          ? 'bg-gradient-to-r from-blue-600 to-indigo-600 text-white shadow-lg shadow-blue-500/30 scale-[1.02]'
                          : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-800 hover:scale-[1.01]'
                      }
                    `}
                  >
                    <div className={`
                      w-8 h-8 rounded-lg flex items-center justify-center font-bold text-xs
                      ${isSelected ? 'bg-white/20 text-white' : colorSet.bg + ' ' + colorSet.text}
                    `}>
                      {category.name.charAt(0).toUpperCase()}
                    </div>
                    <span className="flex-1 text-left truncate">{category.name}</span>
                    {isSelected && (
                      <ChevronRight className="w-4 h-4 text-white/70" />
                    )}
                  </button>
                </li>
              );
            })}
          </ul>
        )}
      </div>

      {/* Footer with count */}
      <div className="px-4 py-2.5 border-t border-gray-200 dark:border-gray-800 bg-gray-50 dark:bg-gray-800/50">
        <p className="text-xs text-gray-500 dark:text-gray-400 text-center">
          {categories.length} {categories.length === 1 ? 'category' : 'categories'}
        </p>
      </div>
    </div>
  );
}
