/**
 * Raw Materials Products Page
 * Display and manage Raw Materials (RM) products
 */

'use client';

import { useState } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { useRouter } from 'next/navigation';
import { productsAPI } from '@/lib/api/products';
import { Button, Card, CardHeader, CardTitle, CardContent, Table, TableHeader, TableBody, TableRow, TableHead, TableCell, Badge, LoadingSpinner, Input } from '@/components/ui';
import { formatCurrency } from '@/lib/utils/format';
import { Plus, Edit, Eye, Box, ArrowLeft, Ruler } from 'lucide-react';
import toast from 'react-hot-toast';

export default function RawMaterialsPage() {
  const router = useRouter();
  const queryClient = useQueryClient();
  const [page, setPage] = useState(1);
  const [limit] = useState(20);
  const [search, setSearch] = useState('');
  const [trackInventoryFilter, setTrackInventoryFilter] = useState('');
  const [dimensionFilter, setDimensionFilter] = useState('');

  // Fetch Raw Materials products only
  const { data, isLoading, error } = useQuery({
    queryKey: ['products', 'RM', page, limit, search, trackInventoryFilter, dimensionFilter],
    queryFn: async () => {
      const params = {
        page,
        limit,
        product_type: 'RM', // Always filter by RM
        ...(search && { search }),
        ...(trackInventoryFilter && { track_inventory: trackInventoryFilter }),
      };
      const response = await productsAPI.list(params);
      return response.data || response;
    },
  });

  const products = data?.products || [];
  const pagination = data?.pagination || {};

  // Filter by dimension tracking on client side (if needed)
  const filteredProducts = dimensionFilter
    ? products.filter((p) => {
        if (dimensionFilter === 'dimension') return p.track_by_dimensions === true;
        if (dimensionFilter === 'quantity') return p.track_by_dimensions === false;
        return true;
      })
    : products;

  const handleSearch = (e) => {
    e.preventDefault();
    setPage(1);
  };

  return (
    <div className="p-6 space-y-6 bg-gray-50 dark:bg-gray-950">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div className="flex items-center gap-4">
          <Button
            variant="outline"
            onClick={() => router.push('/products')}
            className="flex items-center gap-2"
          >
            <ArrowLeft className="w-4 h-4" />
            Back
          </Button>
          <div>
            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Raw Materials</h1>
            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
              Manage raw materials used in production
            </p>
          </div>
        </div>
        <Button
          onClick={() => router.push('/products/new?type=RM')}
          variant="primary"
          className="flex items-center gap-2"
        >
          <Plus className="w-5 h-5" />
          New Raw Material
        </Button>
      </div>

      {/* Filters */}
      <Card>
        <CardContent className="pt-6">
          <form onSubmit={handleSearch} className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="md:col-span-2">
              <Input
                label="Search"
                type="text"
                placeholder="Search by name or SKU..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                Inventory Tracking
              </label>
              <select
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-800 dark:border-gray-600 dark:text-white"
                value={trackInventoryFilter}
                onChange={(e) => {
                  setTrackInventoryFilter(e.target.value);
                  setPage(1);
                }}
              >
                <option value="">All</option>
                <option value="true">Tracked</option>
                <option value="false">Not Tracked</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                Tracking Method
              </label>
              <select
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-800 dark:border-gray-600 dark:text-white"
                value={dimensionFilter}
                onChange={(e) => {
                  setDimensionFilter(e.target.value);
                }}
              >
                <option value="">All Methods</option>
                <option value="dimension">Dimension-based</option>
                <option value="quantity">Quantity-based</option>
              </select>
            </div>
          </form>
        </CardContent>
      </Card>

      {/* Products Table */}
      <Card>
        <CardHeader>
          <CardTitle>Raw Materials ({pagination.total || 0})</CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex justify-center py-12">
              <LoadingSpinner size="lg" />
            </div>
          ) : error ? (
            <div className="text-center py-12 text-red-600 dark:text-red-400">
              Error loading products. Please try again.
            </div>
          ) : filteredProducts.length === 0 ? (
            <div className="text-center py-12 text-gray-500 dark:text-gray-400">
              <Box className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>No raw materials found</p>
              <Button
                variant="primary"
                className="mt-4"
                onClick={() => router.push('/products/new?type=RM')}
              >
                Create Your First Raw Material
              </Button>
            </div>
          ) : (
            <>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Name</TableHead>
                    <TableHead>SKU</TableHead>
                    <TableHead>Tracking Method</TableHead>
                    <TableHead>Unit of Measure</TableHead>
                    <TableHead>Inventory</TableHead>
                    <TableHead align="right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredProducts.map((product) => (
                    <TableRow
                      key={product.id}
                      onClick={() => router.push(`/products/${product.id}`)}
                      className="hover:bg-gray-50 dark:hover:bg-gray-800 cursor-pointer"
                    >
                      <TableCell>
                        <div className="flex items-center gap-3">
                          {product.image_url ? (
                            <img
                              src={product.image_url}
                              alt={product.name}
                              className="w-10 h-10 rounded object-cover"
                            />
                          ) : (
                            <div className="w-10 h-10 rounded bg-blue-100 dark:bg-blue-900 flex items-center justify-center">
                              <Box className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                            </div>
                          )}
                          <div>
                            <div className="font-medium">{product.name}</div>
                            {product.description && (
                              <div className="text-xs text-gray-500 dark:text-gray-400 truncate max-w-xs">
                                {product.description}
                              </div>
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <span className="font-mono text-sm">{product.sku || 'N/A'}</span>
                      </TableCell>
                      <TableCell>
                        {product.track_by_dimensions ? (
                          <Badge variant="info" className="flex items-center gap-1 w-fit">
                            <Ruler className="w-3 h-3" />
                            Dimension-based
                          </Badge>
                        ) : (
                          <Badge variant="default">Quantity-based</Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        {product.track_by_dimensions && product.unit_of_measure ? (
                          <span className="text-sm font-medium">{product.unit_of_measure.toUpperCase()}</span>
                        ) : (
                          <span className="text-gray-400 text-sm">N/A</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {product.track_inventory ? (
                          <Badge variant="success">Tracked</Badge>
                        ) : (
                          <Badge variant="default">Not Tracked</Badge>
                        )}
                      </TableCell>
                      <TableCell align="right">
                        <div className="flex justify-end gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={(e) => {
                              e.stopPropagation();
                              router.push(`/products/${product.id}`);
                            }}
                          >
                            <Eye className="w-4 h-4" />
                          </Button>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={(e) => {
                              e.stopPropagation();
                              router.push(`/products/${product.id}/edit`);
                            }}
                          >
                            <Edit className="w-4 h-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>

              {/* Pagination */}
              {pagination.totalPages > 1 && (
                <div className="mt-4 flex items-center justify-between">
                  <div className="text-sm text-gray-700 dark:text-gray-300">
                    Showing {((page - 1) * limit) + 1} to {Math.min(page * limit, pagination.total)} of {pagination.total} products
                  </div>
                  <div className="flex gap-2">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setPage(p => Math.max(1, p - 1))}
                      disabled={page === 1}
                    >
                      Previous
                    </Button>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setPage(p => Math.min(pagination.totalPages, p + 1))}
                      disabled={page === pagination.totalPages}
                    >
                      Next
                    </Button>
                  </div>
                </div>
              )}
            </>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
