/**
 * Products Hub Page
 * Main products management hub - separate views for Finished Goods and Raw Materials
 */

'use client';

import { useRouter } from 'next/navigation';
import { Card, CardHeader, CardTitle, CardContent, Button } from '@/components/ui';
import { Package, Box, ArrowRight } from 'lucide-react';

export default function ProductsPage() {
  const router = useRouter();

  const menuItems = [
    {
      title: 'Finished Goods',
      description: 'Manage finished products that are sold to customers. These products appear in POS and can be tracked in inventory.',
      icon: Package,
      href: '/products/finished-goods',
      color: 'green',
    },
    {
      title: 'Raw Materials',
      description: 'Manage raw materials used in production. Track by dimensions (fabric, leather) or quantity (zippers, buttons).',
      icon: Box,
      href: '/products/raw-materials',
      color: 'blue',
    },
  ];

  return (
    <div className="p-6 bg-gray-50 dark:bg-gray-950 space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Products</h1>
        <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
          Manage your product catalog - Finished Goods and Raw Materials
        </p>
      </div>

      {/* Menu Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {menuItems.map((item) => {
          const Icon = item.icon;
          return (
            <Card
              key={item.href}
              className="hover:shadow-lg transition-shadow cursor-pointer"
              onClick={() => router.push(item.href)}
            >
              <CardHeader>
                <div className="flex items-center gap-4">
                  <div className={`p-3 rounded-lg bg-${item.color}-100 dark:bg-${item.color}-900`}>
                    <Icon className={`w-6 h-6 text-${item.color}-600 dark:text-${item.color}-400`} />
                  </div>
                  <CardTitle className="text-xl">{item.title}</CardTitle>
                </div>
              </CardHeader>
              <CardContent>
                <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">{item.description}</p>
                <Button
                  variant="outline"
                  className="w-full flex items-center justify-center gap-2"
                  onClick={(e) => {
                    e.stopPropagation();
                    router.push(item.href);
                  }}
                >
                  Open <ArrowRight className="w-4 h-4" />
                </Button>
              </CardContent>
            </Card>
          );
        })}
      </div>
    </div>
  );
}
